-- a session starts with a descriptor
--   . if 0 then assert TFT reset line
--   . else bit 30..0 = session TFT bytes count
--          bit 31 = command+param pair or data only
--
--   if command+param
--     . first byte is TFT byte
--     . second byte indicates whether it's a command or a param

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

entity SDT028ATFTip_v1_0_S00_AXIS is
  generic (
    C_S_AXIS_TDATA_WIDTH  : integer  := 32    -- AXI4Stream sink: Data Width
  );
  port (
    data_out : out std_logic_vector(7 downto 0); -- to tft (bidirectional)
    control : buffer std_logic;
    csx : out std_logic;    -- chip select
    wrx : out std_logic;    -- write
    dcx : out std_logic;    -- data or command
    rdx : out std_logic;    -- read
    resx : out std_logic;   -- reset

    S_AXIS_ACLK  : in std_logic;    -- AXI4Stream sink: Clock
    S_AXIS_ARESETN  : in std_logic; -- AXI4Stream sink: Reset
    S_AXIS_TREADY  : out std_logic; -- Ready to accept data in
    S_AXIS_TDATA  : in std_logic_vector(C_S_AXIS_TDATA_WIDTH-1 downto 0);     -- Data in
    S_AXIS_TSTRB  : in std_logic_vector((C_S_AXIS_TDATA_WIDTH/8)-1 downto 0); -- Byte qualifier
    S_AXIS_TLAST  : in std_logic;   -- Indicates boundary of last packet
    S_AXIS_TVALID  : in std_logic   -- Data is in valid
  );
end SDT028ATFTip_v1_0_S00_AXIS;

architecture arch_imp of SDT028ATFTip_v1_0_S00_AXIS is
  type state is (IDLE, OUT_BYTE, WR_BYTE);
                
  signal tft_state : state;  
  signal axis_tready  : std_logic;
  signal byte_count : std_logic_vector(30 downto 0) := (others => '0');  -- total nb of bytes to transfer
  signal data_buf : std_logic_vector(31 downto 0) := (others => '0');   -- axi 32bit data
  signal byte_pos : std_logic_vector(1 downto 0) := "00"; -- indicates byte position in data_buf
  signal cd_pair : std_logic := '0'; -- indicates cmd/data pair (else data only i.e. pixels)
  
  signal data_tft : std_logic_vector(7 downto 0) := (others => '0');
  signal write_tft : std_logic := '1';
  signal data_cmd_tft : std_logic := '1';
  signal reset_tft : std_logic := '1';
  
begin
  S_AXIS_TREADY  <= axis_tready;
  data_out <= data_tft;
  wrx <= write_tft;
  control <= '1';
  dcx <= data_cmd_tft;
  csx <= '0';
  rdx <= '1';
  resx <= reset_tft;
  
  axis_tready <= '1' when tft_state = IDLE else '0';
  
  process(S_AXIS_ACLK)
  begin
    if rising_edge (S_AXIS_ACLK) then
      if S_AXIS_ARESETN = '0' then
        -- Synchronous reset (active low)
        tft_state <= IDLE;
        write_tft <= '1';
        data_cmd_tft <= '1';
        reset_tft <= '1';
        byte_pos <= "00";
        cd_pair <= '0';
      else
        case tft_state is
          when IDLE =>
            write_tft <= '1';
            data_cmd_tft <= '1';
            byte_pos <= "00";
            if S_AXIS_TVALID = '1' then
              if byte_count = 0 then
                if S_AXIS_TDATA = "00000000000000000000000000000000" then
                  reset_tft <= '0';
                else
                  reset_tft <= '1';
                  byte_count <= S_AXIS_TDATA(30 downto 0); -- first data is session bytes count
                  cd_pair <= S_AXIS_TDATA(31);
                end if;
                tft_state <= IDLE;
              else
                data_buf <= S_AXIS_TDATA; -- 4 bytes buffer
                tft_state <= OUT_BYTE;
              end if;
            else
              tft_state <= IDLE;
            end if;
            
          when OUT_BYTE =>
            case byte_pos is
              when "00" => data_tft <= data_buf(7 downto 0);
                           if cd_pair = '1' then data_cmd_tft <= data_buf(8); else data_cmd_tft <= '1'; end if;
            
              when "01" => data_tft <= data_buf(15 downto 8);
            
              when "10" => data_tft <= data_buf(23 downto 16);
                           if cd_pair = '1' then data_cmd_tft <= data_buf(24); else data_cmd_tft <= '1'; end if;
            
              when "11" => data_tft <= data_buf(31 downto 24);
              
              when others => data_tft <= (others => '0');
            end case;
            
            write_tft <= '0';  -- prepare write
            byte_count <= byte_count - 1;
            tft_state <= WR_BYTE;
                                         
            when WR_BYTE =>
              write_tft <= '1';  -- write on rising edge
              if byte_count = 0 then
                tft_state <= IDLE;
              else
                if cd_pair = '1' then
                  if byte_pos = "10" then tft_state <= IDLE; else tft_state <= OUT_BYTE; end if;
                  byte_pos <= byte_pos + 2;
                else
                  if byte_pos = "11" then tft_state <= IDLE; else tft_state <= OUT_BYTE; end if;
                  byte_pos <= byte_pos + 1;
                end if;
              end if;
         
          when others    => 
            tft_state <= IDLE;
          
        end case;
      end if;  
    end if;
  end process;

end arch_imp;
