/////////////////////////////////////////////////////////////////////////////////////////////
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//
//  *** Interrupt Handling for Xilinx Zynq7000 ***
//
/////////////////////////////////////////////////////////////////////////////////////////////
//#include <tinyhal.h>

#include <stdint.h>
#include <intrinsics.h>
#include "xparameters.h"  // Zynq parameters
#include "Zynq7000_intc.h"
#include "defs.h"

extern UINT32 interrupt_handlers;
static volatile UINT32* _vectors = &interrupt_handlers;

// ---------------------------------------------------------------------------
void CPU_INTC_Initialize() {

  for (int i = 0; i < 5; i++) 
    GIC->ICDICER[i] = 0xffffffff; // disable all interrupts

  for (int i = 0; i < 5; i++) 
    GIC->ICDICPR[i] = 0xffffffff; // clear pending bits

  // Configure priority mask to lowest value (highest number) - enables all priorities 
  GIC->ICCPMR = 0xFF;

  // Enable both secure and non-secure interrupts at CPU interface & distributor
  // Also enable secure id_read & acknowledge of non-secure interrupts   
  GIC->ICCICR = 0x7;
  GIC->ICDDCR = 0x3;  
}

#define MPIDR (__mrc(15, 0, 0, 0, 5) & 0xff)

// ---------------------------------------------------------------------------
// Note: Cortex-A vectors don't pass parameters so ISR_Param is ignored
BOOL CPU_INTC_ActivateInterrupt(UINT32 Irq_Index, CALLBACK_FPN ISR, void* ISR_Param) {

  _vectors[Irq_Index] = (UINT32)ISR;

  GIC->ICCPMR = 0xFF;
  GIC->ICCICR = 1;
  GIC->ICDDCR = 1;
  GIC->ICDISER[Irq_Index/32] |= 1 << (Irq_Index % 32);  // enable IRQ
  GIC->ICDIPR[Irq_Index/4] |= ((1 << 3) << (((Irq_Index % 4) * 8))); 
  if (Irq_Index >= 32)
    GIC->ICDIPTR[(Irq_Index)/4] |= (MPIDR == 1 ? 2 : 1) << ((Irq_Index % 4) * 8);

  return TRUE;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_DeactivateInterrupt(UINT32 Irq_Index) {
  GIC->ICDICER[Irq_Index/32] |= 1 << (Irq_Index % 32);
  return TRUE;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptEnable(UINT32 Irq_Index) {
  BOOL wasEnabled = (GIC->ICDISER[Irq_Index/32] & (1 << (Irq_Index % 32)));
  GIC->ICDISER[Irq_Index/32] |= 1 << (Irq_Index % 32);
  return wasEnabled;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptDisable(UINT32 Irq_Index) {
  BOOL wasEnabled = (GIC->ICDISER[Irq_Index/32] & (1 << (Irq_Index % 32)));
  GIC->ICDICER[Irq_Index/32] |= 1 << (Irq_Index % 32);
  return wasEnabled;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptEnableState(UINT32 Irq_Index) {
  return (GIC->ICDISER[Irq_Index/32] & (1 << (Irq_Index % 32)));
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptState(UINT32 Irq_Index) {
  // Return TRUE if interrupt status is pending.
  return (BOOL)(GIC->ICDISPR[Irq_Index/32] & (1 << (Irq_Index % 32)));
}
