/****************************************************************************
 * DTFT - DisplayTech SDT028ATFT display (ILI9341 TFT controller)
 *
 * File              : dtft_axidma.c
 * This copy created : Oct 11, 2014
 * Version           : 1.1
 * Description       : AXI-DMA drivers to control TFT display
 *
 * MicroZed breakout connections
 * -----------------------------
 * 6 IM0 1      1           0
 * 7 IM1 0      0           0
 * 8 IM2 0      0           0
 * 9 IM3 1      1           0
 *
 * 15 DB17	r19	3   DB0
 * 16 DB16	t19	4   DB1
 * 17 DB15	t11	5   DB2
 * 18 DB14	t12	6   DB3
 * 19 DB13	t10	7   DB4
 * 20 DB12	u12	8   DB5
 * 21 DB11	u13	9   DB6
 * 22 DB10	v12	10  DB7
 *
 * 33 CSX	v13	11
 * 34 WRX	w13	12
 * 35 D/CX	t14	13
 * 36 RDX	p14	14
 * 37 RESX	t15	15
 *
 * 40 VCC		53/54
 * 41 GND		1/2/17/18/59/60
 ***************************************************************************/
#include "defs.h"
#include "ili9341_regs.h"

#include "xparameters.h"
#include "xaxidma.h"

static void XAXIDMA_IRQHandler(int xaxidmaNum);
static void XAXIDMA0_IRQHandler(void* param);
static int XAXIDMA_Transfer(int xaxidmaNum, UINT32 Addr, int Length);

static int reset_tft();
static int sendCmd(UINT8 *cmdlist, int ms);
static UINT32 Dma_Busy(int xaxidmaNum);
static void wait_ms(int ms);

int XAXIDMA_init(int xaxidmaNum);
int tft_init();
int tft_cls(int color);
int window(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1);
int sendData(int color, int len);

extern void Xil_DCacheFlushRange(unsigned int adr, unsigned len);

#define CMDPARAM_PAIR 0x80000000  // bit31 set to 1 to tell the FPGA this is a command/parameter pair list
                                  // the other bits provide the list byte count
                                  // special case: if all 0s, assert the hardware reset line
#define CMD 0     // mark byte as command
#define PARAM 1   // mark byte as parameter
#define EOL 0     // end of list
 
typedef struct {
  XAXIDMA_Type *reg;
  UINT32 irq;
  CALLBACK_FPN isr;
} XAXIDMA_PORT_T;

static XAXIDMA_PORT_T  XAXIDMA_Port[TOTAL_XAXIDMA_PORT] = {
  //                *reg                     irq                                       isr
  { (XAXIDMA_Type *)XPAR_AXI_DMA_0_BASEADDR, XPAR_FABRIC_AXI_DMA_0_MM2S_INTROUT_INTR,  XAXIDMA0_IRQHandler },
};

#define XAXIDMA_REG(x)    (XAXIDMA_Port[x].reg)
#define XAXIDMA_IRQ(x)    (XAXIDMA_Port[x].irq)
#define XAXIDMA_ISR(x)    (XAXIDMA_Port[x].isr)

#define TFT_BUFFER_BASE 0x10000000  // static allocation for now!
static UINT8 *tftBuf = (UINT8 *)TFT_BUFFER_BASE;  // scratch area containing the data to send to the FPGA

// tft command sets
// ----------------
static UINT8 cmdreset[] = { 
  1, CMD_SOFTWARE_RESET,
  EOL
};

static UINT8 cmdInit[] = { 
  1, CMD_DISPLAY_OFF,
  5, CMD_COLUMN_ADDRESS_SET, 0x00, 0x00, 0x00, 0xEF,
  5, CMD_PAGE_ADDRESS_SET, 0x00, 0x00, 0x01, 0x3F,
  2, CMD_COLMOD_PIXEL_FORMAT_SET, 0x55,
  2, CMD_MEMORY_ACCESS_CONTROL, 0x48,
  1, CMD_SLEEP_OUT,
  EOL
};

static UINT8 cmdOn[] = { 
  1, CMD_DISPLAY_ON,
  EOL
};

// Flags interrupt handlers used to notify events to the application context
volatile int TxDone = 1;
volatile int Error = 0;

static void XAXIDMA0_IRQHandler(void* param) { XAXIDMA_IRQHandler(XAXIDMA0); }

// ---------------------------------------------------------------------------
static void XAXIDMA_IRQHandler(int xaxidmaNum) {
  XAXIDMA_Type *xaxidma = XAXIDMA_REG(xaxidmaNum);

  UINT32 IrqStatus = xaxidma->MM2S_DMASR & XAXIDMA_IRQ_ALL_MASK;
  if (!(IrqStatus & XAXIDMA_IRQ_ALL_MASK))
    return;
  xaxidma->MM2S_DMASR = IrqStatus;  // acknowledge pending interrupts

  // If error interrupt is asserted, raise error flag, reset the hardware
  // to recover from the error, and return with no further processing
  if ((IrqStatus & XAXIDMA_IRQ_ERROR_MASK)) {
    Error = 1;
    xaxidma->MM2S_DMACR = XAXIDMA_CR_RESET_MASK;    // reset the dma engine
    // wait for reset completion
    for (int retry = 500; retry > 0; retry--) {
      // reset is done when the reset bit is low
      if (!(xaxidma->MM2S_DMACR & XAXIDMA_CR_RESET_MASK))
        break;
    }
    return;  // Reset should never fail for transmit channel
  }
  // If Completion interrupt is asserted, then set the TxDone flag
  if ((IrqStatus & XAXIDMA_IRQ_IOC_MASK)) {
    TxDone = 1;
  }
}

//----------------------------------------------------------------------
// Routine Name : XAXIDMA_init
// Input        : xaxidmaNum axi dma channel number
// Return       : error condition
// Description  : AXI-DMA initialization
//----------------------------------------------------------------------
int XAXIDMA_init(int xaxidmaNum) {
  XAXIDMA_Type *xaxidma = XAXIDMA_REG(xaxidmaNum);
  int err = RET_ERR;

  for (;;) {
    if ((XPAR_AXI_DMA_0_INCLUDE_MM2S == 0) || (XPAR_AXI_DMA_0_INCLUDE_S2MM != 0) || (XPAR_AXI_DMA_0_INCLUDE_SG != 0))
      break;

    // reset the dma engine so the hardware starts from a known state
    xaxidma->MM2S_DMACR = XAXIDMA_CR_RESET_MASK;

    // wait for reset completion
    int retry;
    for (retry = 500; retry > 0; retry--) {
      // reset is done when the reset bit is low
      if (!(xaxidma->MM2S_DMACR & XAXIDMA_CR_RESET_MASK))
        break;
    }
    if (retry == 0)
      break;  // need system hard reset to recover

    xaxidma->MM2S_DMACR &= XAXIDMA_IRQ_ALL_MASK;   // Disable interrupts
    CPU_INTC_ActivateInterrupt(XAXIDMA_IRQ(xaxidmaNum), XAXIDMA_ISR(xaxidmaNum), 0);
    xaxidma->MM2S_DMACR |= XAXIDMA_IRQ_ALL_MASK;   // enable interrupts

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : XAXIDMA_Transfer
// Input        : xaxidmaNum axi dma channel number
//                BuffAddr the address of the source/destination buffer
//                Length the length of the transfer
// Return       : error condition
// Description  : send to FPGA (one simple transfer submission)
//----------------------------------------------------------------------
static int XAXIDMA_Transfer(int xaxidmaNum, UINT32 Addr, int Length) {
  XAXIDMA_Type *xaxidma = XAXIDMA_REG(xaxidmaNum);
  int err = RET_ERR;

  for (;;) {
    if ((Length < 1) || (Length > XAXIDMA_MAX_TRANSFER_LEN))
      break;

    // wait for last transfer completion
#define TRANSFER_TIMEOUT 1000000  // axis_aclk = 100MHz -> full bitmap transferred in ~3.5ms, 1ms = ~80000 loops
    int retry;
    for (retry = TRANSFER_TIMEOUT; retry > 0; retry--) {
      if (TxDone != 0)
        break;
    }
    if (retry == 0)
      break;

    // if the engine is doing transfer, cannot submit
    if ((!(xaxidma->MM2S_DMASR & XAXIDMA_HALTED_MASK)) && Dma_Busy(xaxidmaNum))
      break;

    UINT32 WordBits = (UINT32)(((unsigned int)XPAR_AXI_DMA_0_M_AXI_MM2S_DATA_WIDTH >> 3) - 1);
    if ((Addr & WordBits) && (!XPAR_AXI_DMA_0_INCLUDE_MM2S_DRE))
      break; // unaligned transfer without DRE
 
    Error = TxDone = 0;

    Xil_DCacheFlushRange(Addr, Length); // flush the source buffer before the DMA transfer
    xaxidma->MM2S_SA = Addr;
    xaxidma->MM2S_DMACR |= XAXIDMA_CR_RUNSTOP_MASK;
    xaxidma->MM2S_LENGTH = Length;  // writing to the BTT register starts the transfer
  
    //while (Dma_Busy(xaxidmaNum))  // in polling mode
    //  ;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

static UINT32 Dma_Busy(int xaxidmaNum) {
  return (XAXIDMA_REG(xaxidmaNum)->MM2S_DMASR & XAXIDMA_IDLE_MASK) ? FALSE : TRUE;
}

//----------------------------------------------------------------------
// tft initialization
int tft_init() {
  int err = RET_ERR;
  for (;;) {
    if (reset_tft() != RET_OK)
      break;
    if (sendCmd(cmdreset, 10) != RET_OK)
      break;
    if (sendCmd(cmdInit, 150) != RET_OK)
      break;
    if (sendCmd(cmdOn, 0) != RET_OK)
      break;
    if (tft_cls(BLACK) != RET_OK)
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// tft controller hardware reset
static int reset_tft() {
  int err = RET_ERR;
  for (;;) {
    *(UINT32 *)tftBuf = 0; // reset low
    if (XAXIDMA_Transfer(XAXIDMA0, (UINT32)tftBuf, 4) != RET_OK)
      break;

    wait_ms(10);

    *(UINT32 *)tftBuf = CMDPARAM_PAIR;  // reset high
    if (XAXIDMA_Transfer(XAXIDMA0, (UINT32)tftBuf, 4) != RET_OK)
      break;

    wait_ms(150);
    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : tft_cls
// Input        : color to fill the screen with
// Return       : error condition
// Description  : Clear the complete screen
//----------------------------------------------------------------------
int tft_cls(int color) {
  int err = RET_ERR;
  for (;;) {
    if (window(1, 1, TFT_WIDTH, TFT_HEIGHT) != RET_OK)
      break;
    if (sendData(color, TFT_WIDTH*TFT_HEIGHT) != RET_OK)
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}
//----------------------------------------------------------------------
// set draw window region
int window(unsigned int x0, unsigned int y0, unsigned int x1, unsigned int y1) {

  UINT8 cmdWindow[] = { 
    1, CMD_NOP,
    5, CMD_COLUMN_ADDRESS_SET, (UINT8)(x0 >> 8), (UINT8)x0, (UINT8)(x1 >> 8), (UINT8)x1, 
    5, CMD_PAGE_ADDRESS_SET, (UINT8)(y0 >> 8), (UINT8)y0, (UINT8)(y1 >> 8), (UINT8)y1,
    1, CMD_MEMORY_WRITE,  // assume next operation will be a GRAM write
    EOL
  };
  return sendCmd(cmdWindow, 0);
}

//----------------------------------------------------------------------
// fill buffer area with given pixel color and send to FPGA
int sendData(int color, int len) {
  UINT8 *dest = tftBuf;
  *(UINT32 *)dest = len*2; // bytes count
  dest += 4;

  for (int i = 0; i < len; i++) {
    *dest++ = color>>8;
    *dest++ = color;
  }
  return XAXIDMA_Transfer(XAXIDMA0, (UINT32)tftBuf, (len*2)+4);  // send to FPGA
}

//----------------------------------------------------------------------
// Routine Name : sendCmd
// Input        : *cmdlist data list to format
//                ms delay to wait after transmission to FPGA
// Return       : error condition
// Description  : format command+param and send to FPGA
//----------------------------------------------------------------------
static int sendCmd(UINT8 *cmdlist, int ms) {
  int len, err;
  UINT8 *destp = tftBuf+4;

  int count = 0;
  for (;;) {
    if ((len = *cmdlist++) == 0)  // end of record list
      break;
    for (int i = 0; i < len; i++) {
      *destp++ = *cmdlist++; // data
      *destp++ = i == 0 ? CMD : PARAM; // command or parameter attribute
      count++;
    }
  }
  if (count & 1) {
    count++; // align on even boundary
    *destp++ = 0;
    *destp++ = 0;
  }
  destp = tftBuf;
  *(UINT32 *)destp = CMDPARAM_PAIR + count;
  count *= 2; // 2 bytes per record
  count += 4; // 1st descriptor
  err = XAXIDMA_Transfer(XAXIDMA0, (UINT32)destp, count);  // send to FPGA
  
  if (ms > 0)
    wait_ms(ms);
  return err;
}

// ---------------------------------------------------------------------------
// not accurate, just some delay
static void wait_ms(int ms) {
  int i, j;
  for (i = 0; i < ms; i++)
    for (j = 0; j < 75000; j++)
      ;
}
