
#include "defs.h"
#include "lpm.h"

static U8 PMU0CF_Local;   // holds the desired Wake-up sources

//----------------------------------------------------------------------
// Routine Name : LPM_Init
// Input        : none
// Return       : none
// Description  : initialize the low power functionality
//----------------------------------------------------------------------
void LPM_Init(void) {
  PMU0CF = WU_CLEAR;
  PMU0CF_Local = 0;
}

//----------------------------------------------------------------------
// Routine Name : LPM_Enable_Wakeup
// Input        : port match
// Return       : none
// Description  : enable wakeup source
//----------------------------------------------------------------------
void LPM_Enable_Wakeup(U8 wakeup) {
  PMU0CF_Local |= (wakeup & PORT_MATCH);
}

//----------------------------------------------------------------------
// Routine Name : LPM_Disable_Wakeup
// Input        : port match
// Return       : none
// Description  : disable wakeup source
//----------------------------------------------------------------------
void LPM_Disable_Wakeup(U8 wakeup) {
  PMU0CF_Local &= ~(wakeup & PORT_MATCH);
}

//----------------------------------------------------------------------
// Routine Name : LPM
// Input        : mode SLEEP or SUSPEND
// Return       : none
// Description  : place the device in low power mode
//----------------------------------------------------------------------
void LPM(U8 mode) {
  U8 CLKSEL_save;
  U8 EA_save;
  U8 PMU0CF_snapshot;
   
  // save current interrupt state and disable interrupts 
  EA_save = EA;
  EA = 0;

  PMU0CF = WU_CLEAR;

  // save current system clock selection and select the low power oscillator
  // divided by 2 as the system clock
  CLKSEL_save = CLKSEL;
  CLKSEL = 0x14;
   
  // enable the Flash read one-shot timer   
  FLSCL &= ~BYPASS;    // clear the one-shot bypass bit
  FLWR  =   NON_ZERO;  // write a "dummy" value to FLWR
   
  // verify that the system clock setting has been applied
  while(!(CLKSEL & 0x80))  // wait until CLKRDY -> 1
    ;
   
  //----------------------------------
  // ~~~ Device in Low Power Mode ~~~
  //
    PMU0CF = (mode | PMU0CF_Local);
  //
  // ~~~   Device is now Awake    ~~~
  //----------------------------------
   
  CLKSEL = CLKSEL_save;  // restore the System Clock
   
  // disable (Bypass) the Flash Read one-shot timer if the system clock
  // frequency is higher than 10 MHz
  if(SYSCLK > 10000000)
    FLSCL |= BYPASS;    // set the one-shot bypass bit                     
  
  // capture the wake-up source and clear all wake-up source flags   
  PMU0CF_snapshot = PMU0CF;
  PMU0CF = CLEAR;

  EA = EA_save;  // restore Interrupt State
}               

