
#include "defs.h"

SBIT (SDA, SFR_P0, 0);
SBIT (SCL, SFR_P0, 1);

#define SMB_FREQUENCY   100000 // target SCL clock rate

#define PCM_ADDR        0x8c   // pcm3052 device address

// status vector - top 4 bits only
#define SMB_MTSTA       0xE0   // master start transmitted
#define SMB_MTDB        0xC0   // master data byte transmitted
#define SMB_MRDB        0x80   // master data byte received

#define SMB_WRITE       0x00
#define SMB_READ        0x01

// ISR state machine
#define START 0
#define ADDR_SEND 1
#define REG_SEND 2
#define REP_START 3
#define DATA_REC 4
#define DATA_SEND 5
#define DATA_DONE 6

static __xdata U8 SMB_REG_ADDR; // global holder for device register address
static __xdata U8 SMB_DATA_IN;  // global holder for SMBus data read
static __xdata U8 SMB_DATA_OUT; // global holder for SMBus data write

static __bit SMB_BUSY; // indicate SMB_Read() or SMB_Write() have claimed the SMBus
static __bit SMB_RW;   // indicate the direction of the current transfer
static __xdata U16 NUM_ERRORS;  // counter for the number of errors.

static void SMBus_Init(void);
static void Timer2_Init(void);
static void Timer3_Init(void);
static void Port_Init(void);
static void SMB_Write(U8, U8);
static U8 SMB_Read(U8);

static __xdata U8 d;

//----------------------------------------------------------------------
// Routine Name : pcm_init
// Input        : none
// Return       : none
// Description  : pcm3052 initialization
//----------------------------------------------------------------------
void pcm_init(void) {
  U8 i;

  // if slave is holding SDA low because of an improper SMBus reset or error
  // provide clock pulses to allow the slave to advance out
  // of its current state. This will allow it to release SDA.
  while (!SDA) {
    XBR2 = 0x40;  // enable Crossbar
    SCL = 0;      // drive the clock low
    for (i = 0; i < 255; i++)  // hold the clock low
      ;
    SCL = 1;      // release the clock
    while (!SCL)  // wait for open-drain clock output to rise
      ;
    for (i = 0; i < 10; i++)   // hold the clock high
      ;
     XBR2 = 0x00; // disable Crossbar
  }
  Port_Init();    // initialize Crossbar and GPIO  
  Timer2_Init();  // configure Timer2 for use as SMBus clock source
  Timer3_Init();  // configure Timer3 for use with SMBus low timeout detect
  SMBus_Init();   // configure and enable SMBus
  
  EIE1 |= 0x01;   // enable the SMBus interrupt
  EA = 1;         // global interrupt enable

  //d = SMB_Read(0x41);
  //d = SMB_Read(0x42);
  //d = SMB_Read(0x43);
  //d = SMB_Read(0x44);
  //d = SMB_Read(0x45);
  //d = SMB_Read(0x46);
  //d = SMB_Read(0x47);
  //d = SMB_Read(0x48);
  //d = SMB_Read(0x4B);
  //d = SMB_Read(0x4D);
  //d = SMB_Read(0x4E);
  //d = SMB_Read(0x4F);
  //d = SMB_Read(0x50);

  SMB_Write(0x48, 0x24);  // mic, pga gain -4dB
  SMB_Write(0x50, 0x21);  // spdif, 24bits
 
  d = SMB_Read(0x48);  // do not comment-out those 2 lines
  d = SMB_Read(0x50);
}

//----------------------------------------------------------------------
// Routine Name : SMBus_Init
// Input        : none
// Return       : none
// Description  : SMBus configured as follows:
//                - SMBus enabled
//                - Slave mode inhibited
//                - Timer0 used as clock source. The maximum SCL frequency
//                  will be approximately 1/3 the Timer0 overflow rate
//                - Setup and hold time extensions enabled
//                - Bus Free and SCL Low timeout detection enabled
//----------------------------------------------------------------------
static void SMBus_Init(void) {

  SMB0CF = 0x5F;  // use Timer2 overflows as SMBus clock source
                  // disable slave mode
                  // enable setup & hold time extensions
                  // enable SMBus Free timeout detect
                  // enable SCL low timeout detect

  SMB0CF |= 0x80; // enable SMBus
  EIP1 = 0x01;    // high priority
}

//----------------------------------------------------------------------
// Routine Name : Timer2_Init
// Input        : none
// Return       : none
// Description  : Timer2 configured as the SMBus clock source as follows:
//                - Timer2 in 8-bit auto-reload mode
//                - SYSCLK or SYSCLK / 4 as Timer0 clock source
//                - Timer2 overflow rate => 3 * SMB_FREQUENCY
//                - The resulting SCL clock rate will be ~1/3 
//                  the Timer2 overflow rate
//                - Timer2 enabled
//----------------------------------------------------------------------
static void Timer2_Init(void) {

// make sure the Timer can produce the appropriate frequency in 8-bit mode
// supported SMBus Frequencies range from 10kHz to 100kHz.  The CKCON register
// settings may need to change for frequencies outside this range.
#if ((SYSCLK/SMB_FREQUENCY/3) < 255)

  #define SCALE 1
  CKCON |= 0x10;    // Timer2 clock source = SYSCLK

#elif ((SYSCLK/SMB_FREQUENCY/4/3) < 255)

  #define SCALE 4
  CKCON |= 0x01;
  CKCON &= ~0x06;   // Timer0 clock source = SYSCLK / 4
#endif

  TMR2RLH   = 0xD7;
  TR2 = TRUE;
}

//----------------------------------------------------------------------
// Routine Name : Timer3_Init
// Input        : none
// Return       : none
// Description  : Timer3 configured for use by the SMBus low timeout detect
//                feature as follows:
//                - Timer3 in 16-bit auto-reload mode
//                - SYSCLK/12 as Timer3 clock source
//                - Timer3 reload registers loaded for a 25ms overflow period
//                - Timer3 pre-loaded to overflow after 25ms
//                - Timer3 enabled
//----------------------------------------------------------------------
static void Timer3_Init(void) {

  TMR3CN = 0x00;  // Timer3 configured for 16-bit auto-reload, low-byte interrupt disabled

  CKCON &= ~0x40; // Timer3 uses SYSCLK/12

  TMR3RL = (unsigned int) -(SYSCLK/12/40);  // Timer3 configured to overflow after
  TMR3 = TMR3RL;                            // ~25ms (for SMBus low timeout detect):
                                            // 1/.025 = 40

  EIE1 |= 0x80;    // Timer3 interrupt enable
  TMR3CN |= 0x04;  // start Timer3
}

//----------------------------------------------------------------------
// Routine Name : portInit
// Input        : none
// Return       : none
// Description  : Configure the Crossbar and GPIO ports
//                SDA                   P0.0  open-drain
//                SCL                   P0.1  open-drain
//----------------------------------------------------------------------
static void Port_Init(void) {

  P0MDOUT = 0x00;                     // All P0 pins open-drain output

  P0SKIP &= ~0b00000011;

  XBR0 |= 0x04;   // enable SMBus
  XBR2 = 0x40;    // enable crossbar and weak pull-ups

  P0 = 0xFF;
}

//----------------------------------------------------------------------
// Routine Name : pcmSMB0_isr
// Input        : none
// Return       : none
// Description  : SMBus Interrupt Service Routine (ISR)
//                SMBus ISR state machine
//                - master only implementation - no slave or arbitration
//                  states defined
//                - all incoming data is written to global var <SMB_DATA_IN>
//                - all outgoing data is read from global var <SMB_DATA_OUT>
//----------------------------------------------------------------------
void pcmSMB0_isr(void) {
  __bit FAIL = 0;             // flag failed transfers
  static U8 state = START;

  if (ARBLOST == 0) {         // check for errors
    switch (SMB0CN & 0xF0) {  // status vector

    // Master Transmitter/Receiver: START condition transmitted.
    case SMB_MTSTA:
      switch (state) {
      case START:
        SMB0DAT = PCM_ADDR & 0xFE;  // device address (R/W bit set to WRITE)
        STA = 0;    // manually clear START bit
        state = ADDR_SEND;
        break;

      case REP_START:
        SMB0DAT = PCM_ADDR | 0x01;  // device address (R/W bit set to READ)
        STA = 0;    // manually clear START bit
        state = DATA_REC;
        break;

      default:
        break;
      }
      break;

    // Master Transmitter: data byte transmitted
    case SMB_MTDB:
      if (ACK) {    // slave ACK?
        switch (state) {
        case ADDR_SEND:
          SMB0DAT = SMB_REG_ADDR; // send register addr
          state = (SMB_RW == SMB_READ) ? REG_SEND : DATA_SEND;
          break;

        case REG_SEND:
          STA = 1;
          state = REP_START;
          break;

        case DATA_SEND:
          SMB0DAT = SMB_DATA_OUT;
          state = DATA_DONE;
          break;

        case DATA_DONE:
          STO = 1;  // set STO to terminate transfer
          SMB_BUSY = 0; // and free SMBus interface
          state = START;
          break;

        case DATA_REC:
          break;

        default:
          STO = 1;  // set STO to terminate transfer
          SMB_BUSY = 0; // and free SMBus interface
          state = START;
          break;
        }
      }
      else {    // if slave NACK,
        STO = 1;  // send STOP condition, followed
        STA = 1;  // by a START
        NUM_ERRORS++;
      }
      break;

    // Master Receiver: data byte received
    case SMB_MRDB:
      // state should be DATA_REC
      SMB_DATA_IN = SMB0DAT;   // store received byte
      SMB_BUSY = 0;  // free SMBus interface
      ACK = 0;    // send NACK to indicate last byte of this transfer

      STO = 1;    // send STOP to terminate transfer
      state = START;
      break;

    default:
      FAIL = 1;   // indicate failed transfer and handle at end of ISR
      state = START;
      break;
    }
  }
  else
    FAIL = 1;   // ARBLOST = 1, error occurred... abort transmission

  if (FAIL) {   // if the transfer failed,
    SMB0CF &= ~0x80;  // reset communication
    SMB0CF |= 0x80;
    STA = 0;
    STO = 0;
    ACK = 0;

    SMB_BUSY = 0;   // free SMBus
    FAIL = 0;
    NUM_ERRORS++;
    state = START;
  }
  SI = 0;   // clear interrupt flag
}

//----------------------------------------------------------------------
// Routine Name : pcmTMR3_isr
// Input        : none
// Return       : none
// Description  : Timer3 Interrupt Service Routine (ISR)
//                a Timer3 interrupt indicates an SMBus SCL low timeout.
//                the SMBus is disabled and re-enabled here
//----------------------------------------------------------------------
void pcmTMR3_isr(void) {

  SMB0CF &= ~0x80;  // disable SMBus
  SMB0CF |= 0x80;   // re-enable SMBus
  TMR3CN &= ~0x80;  // clear Timer3 interrupt-pending flag
  STA = 0;
  SMB_BUSY = 0;     // free SMBus
}

//----------------------------------------------------------------------
// Routine Name : SMB_Write
// Input        : register addr and data to write
// Return       : none
// Description  : writes a single byte to the accelerometer register
//----------------------------------------------------------------------
static void SMB_Write(U8 reg, U8 val) {
   while (SMB_BUSY)     // wait for SMBus to be free
     ;
   SMB_BUSY = 1;        // claim SMBus (set to busy)
   SMB_REG_ADDR = reg;
   SMB_DATA_OUT = val;
   SMB_RW = SMB_WRITE;  // mark this transfer as a WRITE
   STA = 1;             // start transfer
}

//----------------------------------------------------------------------
// Routine Name : SMB_Read
// Input        : register addr
// Return       : value read
// Description  : Reads a single byte from the accelerometer register
//----------------------------------------------------------------------
static U8 SMB_Read(U8 reg) {
  U8 data_in;

  while (SMB_BUSY)    // wait for bus to be free
    ;
  SMB_BUSY = 1;       // claim SMBus (set to busy)
  SMB_REG_ADDR = reg;
  SMB_RW = SMB_READ;  // mark this transfer as a READ

  STA = 1;            // start transfer

  while (SMB_BUSY)    // wait for transfer to complete
    ;

  data_in = SMB_DATA_IN;

  return(data_in);
}
