#include <__cross_studio_io.h>

// Standard includes
#include <limits.h>

#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"

#include "defs.h"
#include "xparameters.h"
#include "wdt.h"

#include "cache.h"
#include "Zynq7000_intc.h"
#include "cpu_locks.h"

// Prototypes for the standard FreeRTOS callback/hook functions implemented within this file
void vApplicationMallocFailedHook(void);
void vApplicationIdleHook(void);
void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName);
void vApplicationTickHook(void);

static void prvSetupHardware(void);
 
#define XSCUGIC_SFI_TRIG_CPU_MASK 0x00FF0000    // CPU Target list 
#define XSCUGIC_SFI_TRIG_INTID_MASK 0x0000000F  // Set to the INTID signaled to the CPU

static void SGI1_IRQHandler();
static void SGI4_IRQHandler();
void interruptSlave(int irqId, int data1, int data2);
void functionCompletion(int cmdId, int retval);

ISR_FN_t interrupt_handlers[160]; 

SGI_PORT_T  SGI_Port[TOTAL_SGI_PORT] = {
  { (volatile LOCK_Type *)LOCK_BASEADDR+0, INTC_SGI1_ID, SGI1_IRQHandler }, // transmitter
  { (volatile LOCK_Type *)LOCK_BASEADDR+1, INTC_SGI2_ID, NULL            }, // receiver
  { (volatile LOCK_Type *)LOCK_BASEADDR+2, INTC_SGI3_ID, NULL            }, // receiver
  { (volatile LOCK_Type *)LOCK_BASEADDR+3, INTC_SGI4_ID, SGI4_IRQHandler }  // SD read access
};

#define DISPATCHER_TASK_PRIORITY (tskIDLE_PRIORITY+1)
extern void dispatcherTask(void *pvParameters);

#define NUM_DISPATCHER_MESSAGES  30
xQueueHandle dispatcherQueue;
CPU0Param dispatcherParam;

static CPU0Param isrFsParam;

int SDisfree = FALSE;
int SDrequested = FALSE;

int main(void) {
  prvSetupHardware();
  dispatcherQueue = xQueueCreate(NUM_DISPATCHER_MESSAGES, sizeof(CPU0Param));
  xTaskCreate(dispatcherTask, (signed char *)"DP", configMINIMAL_STACK_SIZE, NULL, DISPATCHER_TASK_PRIORITY, NULL);
  vTaskStartScheduler();
  for(;;)
    ;
}

//----------------------------------------------------------------------
// Routine Name : SGI1_IRQHandler
// Input        : none
// Return       : none
// Description  : CPU0 to CPU1 ISR
//                put CPU0 params in message queue
//----------------------------------------------------------------------
static void SGI1_IRQHandler() {
  volatile LOCK_Type *lock = SGI_REG(LOCK1);

  while (lock->lock == 1)
    ;
  isrFsParam.cmdId = lock->arg1;
  isrFsParam.bufp = (unsigned char *)lock->arg2;
  isrFsParam.buflen = lock->arg3;
  lock->lock = 1;
  xQueueSendToBackFromISR(dispatcherQueue, &isrFsParam, NULL);
}

//----------------------------------------------------------------------
// Routine Name : SGI4_IRQHandler
// Input        : none
// Return       : none
// Description  : CPU0 to CPU1 ISR
//                cpu0 requesting access to SD read
//----------------------------------------------------------------------
static void SGI4_IRQHandler() {
  volatile LOCK_Type *lock = SGI_REG(LOCK4);

  if (SDisfree == TRUE)
    lock->lock = 0;
  else
    SDrequested = TRUE;
}

//----------------------------------------------------------------------
// Routine Name : interruptSlave
// Input        : irqId   ID to send when generating interrupt
//                data1   optional data1 to send
//                data2   optional data2 to send
// Return       : none
// Description  : generate an InterruptSlave to CPU0
//----------------------------------------------------------------------
void interruptSlave(int irqId, int data1, int data2) {
  volatile LOCK_Type *lock = SGI_REG(LOCK3);

  while (!lock->lock)
    ;
  lock->arg1 = irqId;
  lock->arg2 = data1;
  lock->arg3 = data2;
  lock->lock = 0;

  GIC->ICDSGIR = (UINT32)(TO_CPU0 << 16) | SGI_IRQ(LOCK3);  // send SGI#3 to CPU0
}

//----------------------------------------------------------------------
// Routine Name : functionCompletion
// Input        : cmdId   function ID
//                retval  error status to return
// Return       : none
// Description  : indicates execution completion to CPU0
//----------------------------------------------------------------------
void functionCompletion(int cmdId, int retval) {
  volatile LOCK_Type *lock = SGI_REG(LOCK2);

  while (!lock->lock)
    ;
  lock->arg1 = cmdId;
  lock->arg2 = retval;
  lock->lock = 0;

  GIC->ICDSGIR = (UINT32)(TO_CPU0 << 16) | SGI_IRQ(LOCK2);  // send SGI#2 to CPU0
}


//-----------------------------------------------------------
static void prvSetupHardware(void) {
  // Ensure no interrupts execute while the scheduler is in an inconsistent state.
  // Interrupts are automatically enabled when the scheduler is started
  portDISABLE_INTERRUPTS();
  CPU_INTC_Initialize(XPAR_CPU_ID + 1);
  CPU_INTC_ActivateInterrupt(SGI_IRQ(LOCK1), SGI_ISR(LOCK1), 0);
  CPU_INTC_ActivateInterrupt(SGI_IRQ(LOCK4), SGI_ISR(LOCK4), 0);

  Xil_DCacheEnable();
}

//-----------------------------------------------------------
void vApplicationMallocFailedHook(void) {
  // Called if a call to pvPortMalloc() fails because there is insufficient
  // free memory available in the FreeRTOS heap.  pvPortMalloc() is called
  // internally by FreeRTOS API functions that create tasks, queues, software
  // timers, and semaphores.  The size of the FreeRTOS heap is set by the
  // configTOTAL_HEAP_SIZE configuration constant in FreeRTOSConfig.h
  taskDISABLE_INTERRUPTS();
  for(;;)
    ;
}

//-----------------------------------------------------------
void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName) {
  (void) pcTaskName;
  (void) pxTask;

  // Run time stack overflow checking is performed if
  // configCHECK_FOR_STACK_OVERFLOW is defined to 1 or 2
  for(;;)
    ;
}

//-----------------------------------------------------------
void vApplicationIdleHook(void) {
  volatile size_t xFreeHeapSpace;

  // This is just a trivial example of an idle hook.  It is called on each
  // cycle of the idle task.  It must *NOT* attempt to block.  In this case the
  // idle task just queries the amount of FreeRTOS heap that remains.  See the
  // memory management section on the http://www.FreeRTOS.org web site for memory
  // management options.  If there is a lot of heap memory free then the
  // configTOTAL_HEAP_SIZE value in FreeRTOSConfig.h can be reduced to free up RAM
  xFreeHeapSpace = xPortGetFreeHeapSize();

  // Remove compiler warning about xFreeHeapSpace being set but never used
  (void)xFreeHeapSpace;
}

//-----------------------------------------------------------
void vAssertCalled(const char *pcFile, unsigned long ulLine) {
  volatile unsigned long ul = 0;
  (void) pcFile;
  (void) ulLine;

  taskENTER_CRITICAL();
  {
    // Set ul to a non-zero value using the debugger to step out of this function
    while(ul == 0) {
      portNOP();
    }
  }
  taskEXIT_CRITICAL();
}

//-----------------------------------------------------------
void vApplicationTickHook(void) { }

//---------------------------------------------------------
void vInitialiseTimerForRunTimeStats(void) {
  uint32_t ulValue;
  const uint32_t ulMaxDivisor = 0xff, ulDivisorShift = 0x08;

  ulValue = WDT->CONTROL;
  ulValue |= ulMaxDivisor << ulDivisorShift;
  WDT->CONTROL = ulValue;
  
  WDT->LOAD = UINT_MAX;
  WDT->DISABLE = XSCUWDT_DISABLE_VALUE1;
  WDT->DISABLE = XSCUWDT_DISABLE_VALUE2;
  WDT->CONTROL |= XSCUWDT_CONTROL_WD_ENABLE_MASK;
}
