/////////////////////////////////////////////////////////////////////////////////////////////
//
//  Licensed under the Apache License, Version 2.0 (the "License");
//  you may not use this file except in compliance with the License.
//  You may obtain a copy of the License at http://www.apache.org/licenses/LICENSE-2.0
//
//  Copyright (c) Microsoft Corporation. All rights reserved.
//
//  *** Interrupt Handling for Xilinx Zynq7000 ***
//
/////////////////////////////////////////////////////////////////////////////////////////////
//#include <tinyhal.h>

#include "defs.h"
#include "xparameters.h"  // Zynq parameters

#include <intrinsics.h>
#include "Zynq7000_intc.h"

extern UINT32 interrupt_handlers;
static volatile UINT32* _vectors = &interrupt_handlers;

// ---------------------------------------------------------------------------
void CPU_INTC_Initialize(UINT32 CpuID) {
  UINT32 Int_Id;

#if USE_AMP==1
  #warning "Building GIC for AMP"
  // The distributor should not be initialized by FreeRTOS in the case of AMP
  return;
#endif

  GIC->ICDDCR = 0;

  // Set the security domains in the int_security registers for non-secure interrupts
  // All are secure, so leave at the default. Set to 1 for non-secure interrupts

  // For the Shared Peripheral Interrupts INT_ID[MAX..32], set:

  // 1. The trigger mode in the int_config register
  //    Only write to the SPI interrupts, so start at 32
  for (Int_Id = 32; Int_Id < XSCUGIC_MAX_NUM_INTR_INPUTS; Int_Id += 16) {
    // Each INT_ID uses two bits, or 16 INT_ID per register
    // Set them all to be level sensitive, active HIGH.
    GIC->ICDICFR[Int_Id/16] = 0;
  }

  // 2. The priority using int the priority_level register
  //    The priority_level and spi_target registers use one byte per INT_ID
  //    Write a default value that can be changed elsewhere
#define DEFAULT_PRIORITY    0xa0a0a0a0UL
  for (Int_Id = 0; Int_Id < XSCUGIC_MAX_NUM_INTR_INPUTS; Int_Id += 4) {
    GIC->ICDIPR[Int_Id/4] = DEFAULT_PRIORITY;
  }

  // 3. The CPU interface in the spi_target register
  //    Only write to the SPI interrupts, so start at 32
  //for (Int_Id = 32; Int_Id<XSCUGIC_MAX_NUM_INTR_INPUTS;Int_Id += 4) {
  //  CpuID |= CpuID << 8;
  //  CpuID |= CpuID << 16;
  //  GIC->ICDIPTR[Int_Id/4] = CpuID;
  //}
  // 4. Enable the SPI using the enable_set register. Leave all disabled for now
  for (Int_Id = 0; Int_Id < XSCUGIC_MAX_NUM_INTR_INPUTS; Int_Id += 32) {
    GIC->ICDICER[Int_Id/32] = 0xFFFFFFFFUL;
  }
  GIC->ICDDCR = XSCUGIC_EN_INT_MASK;
  GIC->ICCICR = 0x7;
}

#define MPIDR (__mrc(15, 0, 0, 0, 5) & 0xff) 

// ---------------------------------------------------------------------------
// Note: Cortex-A vectors don't pass parameters so ISR_Param is ignored
BOOL CPU_INTC_ActivateInterrupt(UINT32 Irq_Index, HAL_CALLBACK_FPN ISR, void* ISR_Param) {

  _vectors[Irq_Index] = (UINT32)ISR;

  GIC->ICCPMR = 0xFF;
  GIC->ICCICR = 0x07;
  GIC->ICDDCR = 0x01;
  GIC->ICDISER[Irq_Index/32] |= 1 << (Irq_Index % 32);  // enable IRQ
 if (Irq_Index >= 32) {
    GIC->ICDIPTR[(Irq_Index)/4] &= ~(3 << ((Irq_Index % 4) * 8));  // only 1 cpu destination
    GIC->ICDIPTR[(Irq_Index)/4] |= (MPIDR == 1 ? 2 : 1) << ((Irq_Index % 4) * 8); 

  }
  return TRUE;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_DeactivateInterrupt(UINT32 Irq_Index) {
  GIC->ICDICER[Irq_Index/32] |= 1 << (Irq_Index % 32);
  return TRUE;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptEnable(UINT32 Irq_Index) {
  BOOL wasEnabled = (GIC->ICDISER[Irq_Index/32] & (1 << (Irq_Index % 32)));
  GIC->ICDISER[Irq_Index/32] |= 1 << (Irq_Index % 32);
  return wasEnabled;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptDisable(UINT32 Irq_Index) {
  BOOL wasEnabled = (GIC->ICDISER[Irq_Index/32] & (1 << (Irq_Index % 32)));
  GIC->ICDICER[Irq_Index/32] |= 1 << (Irq_Index % 32);
  return wasEnabled;
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptEnableState(UINT32 Irq_Index) {
  return (GIC->ICDISER[Irq_Index/32] & (1 << (Irq_Index % 32)));
}

// ---------------------------------------------------------------------------
BOOL CPU_INTC_InterruptState(UINT32 Irq_Index) {
  // Return TRUE if interrupt status is pending.
  return (BOOL)(GIC->ICDISPR[Irq_Index/32] & (1 << (Irq_Index % 32)));
}

void CPU_INTC_SetPriorityTriggerType(UINT32 Int_Id, UINT8 Priority, UINT8 Trigger) {
  UINT32 RegValue;

  // Determine the register to write to using the Int_Id.
  RegValue = GIC->ICDIPR[Int_Id/4];

  // The priority bits are Bits 7 to 3 in GIC Priority Register. This
  // means the number of priority levels supported are 32 and they are
  // in steps of 8. The priorities can be 0, 8, 16, 32, 48, ... etc.
  // The lower order 3 bits are masked before putting it in the register.
  Priority = Priority & XSCUGIC_INTR_PRIO_MASK;

  // Shift and Mask the correct bits for the priority and trigger in the register
  RegValue &= ~(XSCUGIC_PRIORITY_MASK << ((Int_Id%4)*8));
  RegValue |= Priority << ((Int_Id%4)*8);

  // Write the value back to the register
  GIC->ICDIPR[Int_Id/4] = RegValue;

  // Determine the register to write to using the Int_Id
  RegValue = GIC->ICDICFR[Int_Id/16];

  // Shift and Mask the correct bits for the priority and trigger in the register
  RegValue &= ~(XSCUGIC_INT_CFG_MASK << ((Int_Id%16)*2));
  RegValue |= Trigger << ((Int_Id%16)*2);

  // Write the value back to the register
  GIC->ICDICFR[Int_Id/16] = RegValue;
}
 

