/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : cdma.c
 * This copy created : May 16, 2015
 * Version           : 1.10
 * Description       : CDMA engine drivers
 ***************************************************************************/
#include <FreeRTOS.h>
#include "task.h"
#include <semphr.h>

#include "defs.h"
#include "xparameters.h"
#include "cache.h"
#include "Zynq7000_intc.h"
#include "cdma.h"

static int errorCounter; // DMA Bus error occurred

extern SemaphoreHandle_t flippedSemaphore;  // tell player flipping of active buffer switch has occurred
extern int activeBufout;  // switch between bufout0 and bufout1

static void cdmaTask(void *pvParameters);
static xQueueHandle cdmaQueue;
static SemaphoreHandle_t cdmaDoneSemaphore;

static int r_id;
static UINT32 destp;
static int destlen;

static void CDMA_IRQHandler(int cdmaNum);
static void CDMA0_IRQHandler(void* param);
  
CDMA_PORT_T CDMA_Port[TOTAL_CDMA_PORT] = {
  { (volatile CDMA_Type *)XPAR_SPDIF_AXI_CDMA_0_BASEADDR, XPAR_FABRIC_SPDIF_AXI_CDMA_0_CDMA_INTROUT_INTR, CDMA0_IRQHandler, FALSE }  // CDMA0
};
  
static void CDMA0_IRQHandler(void* param) { CDMA_IRQHandler(CDMA0); }
static int CDMA_SimpleTransfer(int cdmaNum, int id, UINT32 SrcAddr, UINT32 DstAddr, int Length);

//----------------------------------------------------------------------
// Routine Name : CDMA_IRQHandler
// Input        : cdmaNum = CDMA id number
// Return       : none
// Description  : CDMA interrupt service routine
//----------------------------------------------------------------------
static void CDMA_IRQHandler(int cdmaNum) { 
  UINT32 status, irq;
  BaseType_t xHigherPriorityTaskWoken; 
  volatile CDMA_Type *cdma = CDMA_REG(cdmaNum); 

  status = cdma->SR;
  if ((irq = status & XAXICDMA_XR_IRQ_ALL_MASK) == 0)
    return; // No interrupt for intr handler
  
  cdma->SR = irq;  // acknowledge the interrupt
  CDMA_SimpleNotDone(cdmaNum) = FALSE;
 
  // if has error interrupt, hardware needs to be reset
  if ((irq & XAXICDMA_XR_IRQ_ERROR_MASK) && (status & XAXICDMA_SR_ERR_ALL_MASK)) {
    errorCounter++;
    cdma->CR = XAXICDMA_CR_RESET_MASK;
    for (int timeout = XAXICDMA_RESET_LOOP_LIMIT; timeout > 0; timeout--) {
      if (!(cdma->CR & XAXICDMA_CR_RESET_MASK))
        break;
    }
  }
  if ((r_id & 0xf0) != 0)  // recording
    // invalidate the DestBuffer before receiving the data, in case the Data Cache is enabled
    Xil_DCacheInvalidateRange(destp, destlen);
 
  else  {  // playback
    if ((r_id & 0x07) == 0x07) {
      activeBufout = activeBufout == 0 ? 1 : 0; // flip the switch
      xSemaphoreGiveFromISR(flippedSemaphore, &xHigherPriorityTaskWoken);  // tell player flipping of active buffer switch has occurred
      portYIELD_FROM_ISR(xHigherPriorityTaskWoken);
    }
  }
  xSemaphoreGiveFromISR(cdmaDoneSemaphore, &xHigherPriorityTaskWoken);
  return;
}

//----------------------------------------------------------------------
// Routine Name : CDMA_init
// Input        : cdmaNum = CDMA id number
// Return       : error condition
// Description  : CDMA engine initialization
//----------------------------------------------------------------------
int CDMA_init(int cdmaNum) {
  int timeout, err;
  volatile CDMA_Type *cdma = CDMA_REG(cdmaNum); 

  for (;;) {
    err = RET_ERR;
    errorCounter = 0;

    if ((((unsigned int)XPAR_SPDIF_AXI_CDMA_0_M_AXI_DATA_WIDTH) >> 3) < 4)  // AXI CDMA supports 32 bits data width and up
      break; // word length too short

    CDMA_SimpleNotDone(cdmaNum) = FALSE;  // mark no outstanding transfers

    // Reset the hardware
    cdma->CR = XAXICDMA_CR_RESET_MASK;
    for (timeout = XAXICDMA_RESET_LOOP_LIMIT; timeout > 0; timeout--) {
      if (!(cdma->CR & XAXICDMA_CR_RESET_MASK))
        break;
    }
    if (timeout == 0)
      break; // reset failed

    CPU_INTC_ActivateInterrupt(CDMA_IRQ(cdmaNum), CDMA_ISR(cdmaNum), 0);
    cdma->CR |= XAXICDMA_XR_IRQ_ALL_MASK;  // enable all (completion/error/delay) interrupts
  
    cdmaQueue = xQueueCreate(32, sizeof(CDMA_request));
    xTaskCreate(cdmaTask, (signed char *)"cdma", 1024, NULL, tskIDLE_PRIORITY+1, NULL);
    if ((cdmaDoneSemaphore = xSemaphoreCreateBinary()) == NULL)
      break;
 
    if ((flippedSemaphore = xSemaphoreCreateBinary()) == NULL)
      break;
    
    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : CDMA_SimpleTransfer
// Input        : cdmaNum = CDMA id number
//                id = playback or recording fifo flip
//                srcAddr = source address pointer
//                dstAddr = destination address pointer
//                length = size of the transfer
// Return       : error condition
// Description  : prepares CDMA for a simple transfer
//                if the CDMA is already busy, prepares for transfer next (ISR)
//----------------------------------------------------------------------
static int CDMA_SimpleTransfer(int cdmaNum, int id, UINT32 srcAddr, UINT32 dstAddr, int length) {
  UINT32 WordBits;
  int err;
  volatile CDMA_Type *cdma = CDMA_REG(cdmaNum); 

  for (;;) {
    err = RET_ERR;

    if ((length < 1) || (length > XAXICDMA_MAX_TRANSFER_LEN))
      break;

    WordBits = (UINT32)(((unsigned int)XPAR_SPDIF_AXI_CDMA_0_M_AXI_DATA_WIDTH) >> 3 - 1);
    if ((srcAddr & WordBits) || (dstAddr & WordBits)) {
      if (XPAR_SPDIF_AXI_CDMA_0_INCLUDE_DRE == 0)
        break;  // unaligned transfer without DRE
    }
    // cannot submit if the engine is doing a transfer
    if (!(cdma->SR & XAXICDMA_SR_IDLE_MASK))
      break;
 
    // the driver is still handling the previous simple transfer
    if (CDMA_SimpleNotDone(cdmaNum) == TRUE)
      break;

    r_id = id;
    if ((id & 0xf0) != 0) {
      // save a copy of destination address and length in order to invalidate the cache in the isr
      destp = dstAddr;
      destlen = length;
    }
    else {
      // flush the srcBuffer before the DMA transfer, in case the Data Cache is enabled
      Xil_DCacheFlushRange(srcAddr, length);
    }
    // setup the flag so that others will not step on us (only set if the system is in interrupt mode)
    if ((cdma->CR & XAXICDMA_XR_IRQ_SIMPLE_ALL_MASK) != 0x0)
      CDMA_SimpleNotDone(cdmaNum) = TRUE;

    cdma->SRCADDR = srcAddr;
    cdma->DSTADDR = dstAddr;
    cdma->BTT = length;  // writing to the BTT register starts the transfer

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : CDMA_transfer
// Input        : id = playback or recording fifo flip
//                srcAddr = source address pointer
//                dstAddr = destination address pointer
//                length = size of the transfer
//                isISR = invoqued from ISR or not
// Return       : none
// Description  : post a cdma transfer request in queue
//                if not from ISR then post in front of queue else back
//----------------------------------------------------------------------
void CDMA_transfer(int id, UINT32 srcAddr, UINT32 dstAddr, int length, int isISR) {
  CDMA_request cdmaRequest;

  cdmaRequest.id = id;
  cdmaRequest.srcAddr = srcAddr;
  cdmaRequest.destAddr = dstAddr;
  cdmaRequest.len =  length;
  if (isISR == TRUE)
    xQueueSendToBackFromISR(cdmaQueue, &cdmaRequest, NULL);
  else
    xQueueSendToFront(cdmaQueue, &cdmaRequest, NULL);
}

//----------------------------------------------------------------------
// Routine Name : cdmaTask
// Input        : pvParameters = unused
// Return       : none
// Description  : wait on queue for cdma request then wait for cdma completion
//----------------------------------------------------------------------
static void cdmaTask(void *pvParameters) {
  CDMA_request cdmaRequest;

  for( ;; ) {
    xQueueReceive(cdmaQueue, &cdmaRequest, portMAX_DELAY);
    CDMA_SimpleTransfer(CDMA0, cdmaRequest.id, cdmaRequest.srcAddr, cdmaRequest.destAddr, cdmaRequest.len);
    xSemaphoreTake(cdmaDoneSemaphore, portMAX_DELAY);
  }
}
