/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : convolver.cpp
 * This copy created : Apr 19, 2015
 * Version           : 1.11
 * Description       : convolver engine
 *                     allows:
 *                       . convolution with DRC file
 *                       . convolution with inverse logsweep
 *                     originated from Gian-Carlo Pascutto (foobar2000 convolver)
 ***************************************************************************/
#include "defs.h"
#include <math.h>

#include "wavefile.h"
#include "convolver.h"
#include "play.h"

static void convolveChunk(float *in1_p, float *in2_p, float *out_p, float *lo_p, int nSamples);
static void calcFFT(float *src, float *dest1, float *dest2, int nSamples);
static void clearBuffer(float *bufp, int nSamples);

extern "C" {
  void cdft(int, int, float *);
}

static float *temp_p = (float *)BUF_TEMP;
static float *signal_p = (float *)BUF_SIGNAL;
static float *convsignal_p = (float *)BUF_CONVSIGNAL;
static float *lobuf_p = (float *)BUF_LO;

static float scale_coef, scale_coef2;
static int nSamples;
static int thisType;
 
//----------------------------------------------------------------------
// Routine Name : convolverInit
// Input        : nbSamples = size of convolution
// Return       : none
// Description  : initializes the convolver
//----------------------------------------------------------------------
void convolverInit(int nbSamples) {
  nSamples = nbSamples;
  clearBuffer(temp_p, nSamples*2);
  clearBuffer(signal_p, nSamples*2);
  clearBuffer(lobuf_p, nSamples*2);
  scale_coef = 1.0f / ((float)(nSamples*2));
}

//----------------------------------------------------------------------
// Routine Name : convolverImpulse
// Input        : type = generate data or read from file (WAVEFILE or INVLOGSWEEP)
//                filename = WAVEFILE name (DRC)
// Return       : error condition
// Description  : prepare the convolver with the impulse to multiply to data
//----------------------------------------------------------------------
int convolverImpulse(SourceType type, char *filename) {
  int err;

  thisType = type;
  for (;;) {
    err = RET_ERR;

    play_open(type, filename);  // either WAVEFILE (i.e.drc.wav) or INVLOGSWEEP (i.e. no file)
    int rSamples = nSamples * (play_getTrack() == MONO ? 1 : 2);
    if (play_read(temp_p, rSamples) != rSamples)
      break;
    float *destp2 = (play_getTrack() == MONO ? NULL : convsignal_p + (nSamples * 4));
    calcFFT(temp_p, convsignal_p, destp2, nSamples);  // convert to frequency domain

    err = RET_OK;
    break;  // exit the forever loop
  }
  play_close();
  return err;
}

//----------------------------------------------------------------------
// Routine Name : convolverScaleCoef2
// Input        : cscaleCoef2 = 2nd scale coefficient
// Return       : none
// Description  : attenuation to avoid clipping at the convolution stage
//----------------------------------------------------------------------
void convolverScaleCoef2(float cscaleCoef2) {
  scale_coef2 = cscaleCoef2;
}

//----------------------------------------------------------------------
// Routine Name : convolve
// Input        : bufp = audio data buffer pointer
//                       will contain the convolved data after convolution
// Return       : none
// Description  : do the convolution
//----------------------------------------------------------------------
void convolve(float *bufp) {
  switch (thisType) {
  case WAVEFILE:
    calcFFT(bufp, signal_p, signal_p + (nSamples * 4), nSamples);
    convolveChunk(signal_p, convsignal_p, temp_p, lobuf_p, nSamples);  // left channel
    convolveChunk(signal_p + (nSamples * 4), convsignal_p + (nSamples * 4), temp_p + (nSamples * 4), lobuf_p + (nSamples * 4), nSamples);  // right channel

    float *leftp, *rightp;
    leftp = temp_p;
    rightp = temp_p + (nSamples * 4);
    for (int i = 0; i < nSamples; i++) {
      *bufp++ = *leftp++;
      *bufp++ = *rightp++;
    }
    break;

  case INVLOGSWEEP:
    calcFFT(bufp, signal_p, NULL, nSamples);  // convert measurement to frequency domain
    convolveChunk(signal_p, convsignal_p, bufp, lobuf_p, nSamples);
    clearBuffer(signal_p, nSamples);
    convolveChunk(signal_p, convsignal_p, bufp+nSamples, lobuf_p, nSamples);
    break;
  }
}

//----------------------------------------------------------------------
// Routine Name : convolveChunk
// Input        : in1_p = audio data input #1 buffer pointer
//                in2_p = impulse input #2 buffer pointer
//                out_p = result buffer pointer
//                lo_p = overlap buffer pointer from one chunk to another
//                nSamples = number of samples to convolve
// Return       : none
// Description  : do a convolution on a chunk of data
//----------------------------------------------------------------------
static void convolveChunk(float *in1_p, float *in2_p, float *out_p, float *lo_p, int nSamples) {
  float c1, c2, r1, r2;
  float *bufp, *lobufp;

  // impulse multiply
  bufp = out_p;
  for (int i = 0; i < 2*nSamples; i++) {
    r1 = *in1_p++;
    c1 = *in1_p++;
    r2 = *in2_p++;
    c2 = *in2_p++;

    *bufp++ = r1*r2-c1*c2;
    *bufp++ = c1*r2+r1*c2;
  }
  cdft(4*nSamples, -1, out_p);  // inverse fft

  bufp = out_p;
  lobufp = lo_p;
  for (int i = 0; i < nSamples; i++)
    bufp[i] = (*lobufp++ + (bufp[i<<1] * scale_coef)) * scale_coef2;

  // copy to lobuf
  bufp = out_p+(nSamples*2);
  lobufp = lo_p;
  for (int i = 0; i < nSamples; i++) {
    *lobufp++ = *bufp++ * scale_coef;
    bufp++;
  }
}

//----------------------------------------------------------------------
// Routine Name : calcFFT
// Input        : src = source buffer pointer
//                dest1 = destination #1 buffer pointer
//                dest2 = destination #2 buffer pointer (not NULL if stereo)
//                nSamples = buffer size
// Return       : none
// Description  : prepare buffers et do a FFT
//----------------------------------------------------------------------
static void calcFFT(float *src, float *dest1, float *dest2, int nSamples) {
  float *bufp1, *bufp2;

  bufp1 = dest1;
  bufp2 = dest2;
  if (dest2 == NULL) {
    // mono
    for (int i = 0; i < nSamples; i++) {
      *bufp1++ = *src++;
      *bufp1++ = 0.0;
    }
    for (int i = 0; i < nSamples; i++) {
      *bufp1++ = 0.0;
      *bufp1++ = 0.0;
    }
    cdft(4*nSamples, 1, dest1);
  }
  else {
    // stereo
    for (int i = 0; i < nSamples; i++) {
      *bufp1++ = *src++;
      *bufp1++ = 0.0;
      *bufp2++ = *src++;
      *bufp2++ = 0.0;
    }
    for (int i = 0; i < nSamples; i++) {
      *bufp1++ = 0.0;
      *bufp1++ = 0.0;
      *bufp2++ = 0.0;
      *bufp2++ = 0.0;
    }
    cdft(4*nSamples, 1, dest1);
    cdft(4*nSamples, 1, dest2);
  }
}

//----------------------------------------------------------------------
// Routine Name : clearBuffer
// Input        : bufp = buffer pointer
//                nSamples = buffer size
// Return       : none
// Description  : clear FFT buffer
//----------------------------------------------------------------------
static void clearBuffer(float *bufp, int nSamples) {
  for (int i = 0; i < nSamples; i++) {
    *bufp++ = 0.0;
    *bufp++ = 0.0;
    *bufp++ = 0.0;
    *bufp++ = 0.0;
  }
}
