/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : fifo.c
 * This copy created : May 18, 2015
 * Version           : 1.14
 * Description       : fifo communication PL/PS for proper CDMA BRAM transfer
 ***************************************************************************/
#include "defs.h"
#include "xparameters.h"  // Zynq parameters
#include "Zynq7000_intc.h"
#include "fifo.h"
#include "cdma.h"

#define WORD_SIZE 4	    // fifo: size of words in bytes

static int errorCounter;  // FIFO error counter set in isr

// complete spdif_in data is in dram (writing to SD during spdif in/out crashes the system !!) 
extern short *recordBuf;

extern int activeBufout;  // switch between bufout0 and bufout1
extern short *shortBufout0;
extern short *shortBufout1;
extern short *blankBuf;
extern int exitPlay;

// fpga bram blocks
static UINT32 bramOUT_0 = XPAR_BRAM_0_BASEADDR;
static UINT32 bramOUT_1 = XPAR_BRAM_1_BASEADDR;
static UINT32 bramIN_0 = XPAR_BRAM_2_BASEADDR;
static UINT32 bramIN_1 = XPAR_BRAM_3_BASEADDR;

static uint32 lastbramFlip[2];

static void FIFO_IRQHandler(int fifoNum);
static void FIFO0_IRQHandler(void* param) { FIFO_IRQHandler(FIFO0); }
static void FIFO1_IRQHandler(void* param) { FIFO_IRQHandler(FIFO1); }

FIFO_PORT_T  FIFO_Port[TOTAL_FIFO_PORT] = {
  { (volatile FIFO_Type *)XPAR_AXI_FIFO_0_BASEADDR, XPAR_FABRIC_SPDIF_AXI_FIFO_MM_S_0_INTERRUPT_INTR, FIFO0_IRQHandler }, // FIFO0
  { (volatile FIFO_Type *)XPAR_AXI_FIFO_1_BASEADDR, XPAR_FABRIC_SPDIF_AXI_FIFO_MM_S_1_INTERRUPT_INTR, FIFO1_IRQHandler }  // FIFO1
};

// spdif out bram0/1 = 8192 * 64bit = 65536 bytes = 16384 stereo lp/hp samples
// 8 * bram = 524288 bytes = 131072 stereo lp/hp samples
// a18/a17/a16 (= bit2/bit1/bit0)
//   0   0   0	bram0 = part1
//   0   0   1	bram1 = " 2
//   0   1   0	bram0 = " 3
//   0   1   1	bram1 = " 4
//   1   0   0	bram0 = " 5
//   1   0   1	bram1 = " 6
//   1   1   0	bram0 = " 7
//   1   1   1	bram1 = " 8

// spdif in bram2/3 = 4096 * 64bit = 32768 bytes = 16384 mono samples

//----------------------------------------------------------------------
// Routine Name : FIFO_IRQHandler
// Input        : fifoNum = fifo id number
// Return       : none
// Description  : fifo interrupt service routine
//                when the FPGA is done with a BRAM block (either spdif in or spdif out)
//                it writes to the fifo which generates an interrupt to the PS in order
//                to have the next block CMDA'ed from/to the PL.
//                meanwhile the FPGA is outputting or inputting from/to the other BRAM block
//----------------------------------------------------------------------
static void FIFO_IRQHandler(int fifoNum) {
  volatile FIFO_Type *fifo = FIFO_REG(fifoNum);
  uint32 ReceiveLength, Pending;
  uint32 bramFlip;
  short *src;

  Pending = fifo->IER & fifo->ISR;
  while (Pending) {
    if (Pending & XLLF_INT_RC_MASK) {
      ReceiveLength = fifo->RLF/WORD_SIZE;
      for (int i = 0; i < ReceiveLength; i++)
        bramFlip = fifo->RDFD;

      if (lastbramFlip[fifoNum] != -1) {  // ignore if not initialized
        if (bramFlip != lastbramFlip[fifoNum]) {
          switch (fifoNum) {
          case FIFO0:  // spdif out
            if (exitPlay == TRUE)
              src = blankBuf;  // substitute with empty buffer
	    else {
              // cdma writes from shortbufout0 when activeBufout = 0
              // -> when activeBufout = 0 then copy SD to shortbufout1 inwaveOut()
              src = activeBufout == 0 ? shortBufout0 : shortBufout1;
              src += (BRAM_OUT__DEPTH * (BRAM_WIDTH/16) * (bramFlip & 0x07));  // src in number of words
            }
            // fpga reads from bramOut_0 when a16 = 0 (a16 = bramFlip lower bit)
            // -> when a16 flips to 1 then FIFO_IRQHandler does cdma transfer of ddr to bramOut_0 (actually the other way around !!!)
            CDMA_transfer(bramFlip & 0x0f, (UINT32)src, (bramFlip & 1) == 1 ? bramOUT_1 : bramOUT_0,  BRAM_OUT__DEPTH*(BRAM_WIDTH/8), TRUE);
            break;

          case FIFO1:  // spdif in
            // fpga writes to bramIn_0 when a14 = 0
            // -> when a14 flips to 1 then FIFO_IRQHandler does cdma transfer of bramIn_0 to ddr
            if (recordBuf - (short *)BUF_RECORD >= BUF_RECORD_SIZE)  // circular buffer
              recordBuf = (short *)BUF_RECORD;
            CDMA_transfer(bramFlip & 0xf0, (bramFlip & 0x10) == 0x10 ? bramIN_0 : bramIN_1, (UINT32)recordBuf, BRAM_IN_DEPTH*(BRAM_WIDTH/8), TRUE);
            recordBuf += (BRAM_IN_DEPTH * (BRAM_WIDTH/16));  // dest in number of words
            break;
          }
        }
      }
      lastbramFlip[fifoNum] = bramFlip;
      fifo->ISR = XLLF_INT_RC_MASK & XLLF_INT_ALL_MASK;
    }
    else if (Pending & XLLF_INT_ERROR_MASK){
      if (Pending & XLLF_INT_RPURE_MASK) {
        fifo->RDFR = XLLF_RDFR_RESET_MASK;
      } else if (Pending & XLLF_INT_RPORE_MASK) {
        fifo->RDFR = XLLF_RDFR_RESET_MASK;
      } else if(Pending & XLLF_INT_RPUE_MASK) {
        fifo->RDFR = XLLF_RDFR_RESET_MASK;
      } else if (Pending & XLLF_INT_TPOE_MASK) {
        fifo->TDFR = XLLF_TDFR_RESET_MASK;
      } else if (Pending & XLLF_INT_TSE_MASK) {
      } 
      errorCounter++;
      fifo->ISR = XLLF_INT_ERROR_MASK & XLLF_INT_ALL_MASK;
    }
    else {
      fifo->ISR = Pending & XLLF_INT_ALL_MASK;
    }
    Pending = fifo->IER & fifo->ISR;
  }
}

//----------------------------------------------------------------------
// Routine Name : FifoInit
// Input        : fifoNum = fifo id number
// Return       : error condition
// Description  : fifo initialization
//----------------------------------------------------------------------
int FifoInit(int fifoNum) {
  volatile FIFO_Type *fifo = FIFO_REG(fifoNum);
  int status;

  errorCounter = 0;
  lastbramFlip[fifoNum] = -1;

  fifo->TDFR = XLLF_TDFR_RESET_MASK;
  fifo->RDFR = XLLF_RDFR_RESET_MASK;

  // Reset the core and generate the external reset by writing to
  // the Local Link/AXI Streaming Reset Register
  fifo->LLR = XLLF_RDFR_RESET_MASK;
 
  // Check for the Reset value
  status = fifo->ISR;
  fifo->ISR = XLLF_INT_ALL_MASK;
  status = fifo->ISR;
  if(status != 0)
    return RET_ERR;
    
  CPU_INTC_ActivateInterrupt(FIFO_IRQ(fifoNum), FIFO_ISR(fifoNum), 0);

  return RET_OK;
}
