#include "defs.h"
#include "xparameters.h"
#include "i2c.h"

#define SEND           1  // Transfer direction is sending
#define RECEIVE        0  // Transfer direction is receiving
#define I2C_SCLK_RATE  100000

static void I2C0_IRQHandler(void* param);
static void I2C_IRQHandler(int i2cNum);

static void reset();
static int setSClk(UINT32 FsclHz);
static int setupMaster(int role);

I2C_PORT_T I2C_Port[TOTAL_I2C_PORT] = {
 { (volatile I2C_Type *)XPAR_XIICPS_0_BASEADDR, XPAR_XIICPS_0_INTR, I2C0_IRQHandler }
};

static void I2C0_IRQHandler(void* param) { I2C_IRQHandler(I2C0); }

static void I2C_IRQHandler(int i2cNum) {
  volatile I2C_Type *i2c = I2C_REG(i2cNum);
}

int i2c_init() {
  reset();
  return setSClk(I2C_SCLK_RATE);
}

int i2c_write(UINT16 addr, UINT8 *bufp, int len) {
  volatile I2C_Type *i2c = I2C_REG(I2C0);
  int err;

  for (;;) {
    err = RET_ERR;
    UINT32 Intrs = I2C_IXR_ARB_LOST_MASK | I2C_IXR_TX_OVR_MASK | I2C_IXR_NACK_MASK;

    if (len > I2C_FIFO_DEPTH)
      break;  // for now do not allow chunks bigger than Fifo size

    if (setupMaster(SEND) != RET_OK)
      break;

    i2c->ISR = i2c->ISR;  // clears the interrupt status register
    for (int i = 0; i < len; i++)
      i2c->DATA = *bufp++;  // fills the Fifo
    i2c->ADDR = addr;  // initiates the I2C transfer

    // wait for completion of transfer
    while ((i2c->ISR & I2C_IXR_COMP_MASK) == 0) {
      if ((i2c->ISR & Intrs) != 0)
          break;
    }
    if ((i2c->ISR & Intrs) != 0)
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

int i2c_read(UINT16 addr, UINT8 *bufp, int len) {
  volatile I2C_Type *i2c = I2C_REG(I2C0);
  int err;

  for (;;) {
    err = RET_ERR;
    UINT32 Intrs = I2C_IXR_ARB_LOST_MASK | I2C_IXR_RX_OVR_MASK | I2C_IXR_RX_UNF_MASK | I2C_IXR_NACK_MASK;

    if (len > I2C_DATA_INTR_DEPTH)
      break;  // for now do not allow chunks bigger than Fifo size

    if (setupMaster(RECEIVE) != RET_OK)
      break;

    i2c->ISR = i2c->ISR;  // clears the interrupt status register
    i2c->TRANS_SIZE = len;
    i2c->ADDR = addr;  // initiates the I2C transfer

    while ((len > 0) && ((i2c->ISR & Intrs) == 0)) {
      if ((i2c->SR & I2C_SR_RXDV_MASK) != 0) {
	*bufp++ = i2c->DATA;
        len--;
      }
    }
    // wait for completion of transfer
    while ((i2c->ISR & I2C_IXR_COMP_MASK) == 0) {
      if ((i2c->ISR & Intrs) != 0)
        break;
    }
    if ((i2c->ISR & Intrs) != 0)
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

/*****************************************************************************
* sets the serial clock rate for the I2C device
* the device must be idle before setting these device options
*
* the data rate formula for determining the correct register values is:
* Fscl = Fpclk/(22 x (divisor_a+1) x (divisor_b+1))
* (the clock can not be faster than the input clock divide by 22)
*
* FsclHz  clock frequency in Hz (100KHz and 400KHz are the two most common)
* return  error status
******************************************************************************/
static int setSClk(UINT32 FsclHz) {
  volatile I2C_Type *i2c = I2C_REG(I2C0);
  int err;
  UINT32 Div_a, Div_b;
  UINT32 ActualFscl;
  UINT32 Temp, TempLimit;
  UINT32 LastError, BestError, CurrentError;
  UINT32 ControlReg;
  UINT32 CalcDivA, CalcDivB;
  UINT32 BestDivA = 0;
  UINT32 BestDivB = 0;

  for (;;) {
    err = RET_ERR;
  
    if (i2c->TRANS_SIZE != 0)
      break;

    // assume Div_a is 0 and calculate (divisor_a+1) x (divisor_b+1)
    if ((Temp = XPAR_XIICPS_0_I2C_CLK_FREQ_HZ / (22 * FsclHz)) == 0)
      break;

    // due to a hardware limitation:
    // 384.6KHz should be set if frequency 400KHz is selected
    // 90KHz should be set if frequency 100KHz is selected
    if (FsclHz > 384600)
      FsclHz = 384600;
    if ((FsclHz <= 100000) && (FsclHz > 90000))
      FsclHz = 90000;

    // TempLimit helps in iterating over the consecutive value of Temp to
    // find the closest clock rate achievable with divisors.
    // Iterate over the next value only if fractional part is involved
    TempLimit = (XPAR_XIICPS_0_I2C_CLK_FREQ_HZ % (22 * FsclHz)) ? Temp + 1 : Temp;
    BestError = FsclHz;
    for ( ; Temp <= TempLimit; Temp++) {
      LastError = FsclHz;
      CalcDivA = CalcDivB = CurrentError = 0;

      for (Div_b = 0; Div_b < 64; Div_b++) {
        Div_a = Temp / (Div_b + 1);
        if (Div_a != 0)
          Div_a = Div_a - 1;
        if (Div_a > 3)
          continue;

        ActualFscl = XPAR_XIICPS_0_I2C_CLK_FREQ_HZ / (22 * (Div_a + 1) * (Div_b + 1));
        CurrentError = ActualFscl > FsclHz ? ActualFscl - FsclHz : FsclHz - ActualFscl;
        if (LastError > CurrentError) {
          CalcDivA = Div_a;
          CalcDivB = Div_b;
          LastError = CurrentError;
        }
      }
      // used to capture the best divisors
      if (LastError < BestError) {
        BestError = LastError;
        BestDivA = CalcDivA;
        BestDivB = CalcDivB;
      }
    }
    // read the control register and mask the Divisors
    ControlReg = i2c->CR & ~(I2C_CR_DIV_A_MASK | I2C_CR_DIV_B_MASK);
    ControlReg |= (BestDivA << I2C_CR_DIV_A_SHIFT) | (BestDivB << I2C_CR_DIV_B_SHIFT);
    i2c->CR = ControlReg;

    err = RET_OK;
    break;
  }
}

// reset sequence to the i2c interface 
static void reset() {
  volatile I2C_Type *i2c = I2C_REG(I2C0);

  i2c->IDR = I2C_IXR_ALL_INTR_MASK;  // disable all the interrupts
  i2c->CR = I2C_CR_RESET_VALUE;
  i2c->ISR = i2c->ISR;  // clear the interrupt status
  i2c->TIME_OUT = I2C_TO_RESET_VALUE;
  i2c->TRANS_SIZE = 0x0;
  i2c->SR = i2c->SR;  // clear the status register
}

/*****************************************************************************
* prepares a device to transfer as a master
* role   specifies whether the device is sending or receiving
* return error status
****************************************************************************/
static int setupMaster(int role) {
  volatile I2C_Type *i2c = I2C_REG(I2C0);

  UINT32 ControlReg = i2c->CR;
  if (((ControlReg & I2C_CR_HOLD_MASK) == 0) && (i2c->SR & I2C_SR_BA_MASK))  // check if bus is busy
    return RET_ERR;
 
  // set up master, AckEn, nea and clear fifo
  ControlReg |= I2C_CR_ACKEN_MASK | I2C_CR_CLR_FIFO_MASK | I2C_CR_NEA_MASK | I2C_CR_MS_MASK;
  if (role == RECEIVE)
    ControlReg |= I2C_CR_RD_WR_MASK;
  else
    ControlReg &= ~I2C_CR_RD_WR_MASK;
  i2c->CR = ControlReg;

  return RET_OK;
}
