/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : ir.cpp
 * This copy created : Apr 18, 2015
 * Version           : 1.4
 * Description       : convert the .wav file audio data into impulse response
 ***************************************************************************/
 #include "defs.h"

#include <math.h>
#include "string.h"
#include "fatfs.h"

#include "wavefile.h"
#include "ir.h"
#include "convolver.h"
#include "play.h"

#define IR_SIZE 131072

typedef struct {
  Wavefile *wave;  
  long curpos;
  long crosspos;
  float curSample;
  float crossSample;
  float lastSample;
  float buf[FRAMES_PER_BLOCK * SAMPLES_PER_FRAME];
  unsigned int bufindex;
  unsigned int bufsize;
} getsample;
static getsample gs_desc;

static int locateSinewave(char *, int, int);
static int getCountSamples(getsample *desc, float minlevel);
static getsample *getsampleOpen(char *);
static float getsampleRead(getsample *, bool);
static void getsampleClose(getsample *);

static float *calib_p = (float *)BUF_CALIB;

//----------------------------------------------------------------------
// Routine Name : ir
// Input        : filename = .wav input filename (non-LFN 8 char + extension)
//                nSamples = number of samples to convert
// Return       : error condition
// Description  : convert into impulse response
//----------------------------------------------------------------------
int ir(char *filename, int nSamples) {
  int pos;
  Wavefile *in = NULL;
  FIL file;
  int err;

  for (;;) {
    err = RET_ERR;

    // locate the sinewave chunk in the measurement audio data
    if ((pos = locateSinewave(filename, SAMPLING_RATE, SINEWAVE_FP_FREQ)) < 0)
      break;

    in = new Wavefile();
    if (in->open(filename, FILE_READ) != RET_OK)
      break;

    // move to right after the sinewave chunk
    in->Pos(pos);
    in->seek();

    // read the measurement and convert to float
    if (in->read(calib_p, nSamples) != nSamples)
      break;

    // clear loudspeaker ripples after sinewave (empirical number of samples)
    for (int i = 0; i < 7000; i++)
      calib_p[i] = 0.0;

    convolverInit(nSamples);
    convolverScaleCoef2(728e-8);
    if (convolverImpulse(INVLOGSWEEP, (char *)"") != RET_OK)
      break;
    convolve(calib_p);

    // save result to IR file 32-bit IEEE Float (0.24)
    char name[16];  // Fatfs not configured for LFN
    strcpy(name, filename);
    name[0] = 'i';
    name[1] = 'r';
    int len = strlen(name);
    name[len-3] = 'p';
    name[len-2] = 'c';
    name[len-1] = 'm';

    unsigned int bw;
    if (f_open(&file, name, FA_CREATE_ALWAYS | FA_WRITE) != FR_OK)
      break;
    if (f_write(&file, calib_p, nSamples*2*sizeof(float), &bw) != FR_OK)
      break;
    f_close(&file);

    err = RET_OK;
    break;  // exit the forever loop
  }
  if (in != NULL)
    delete in;
  return err;
}

//----------------------------------------------------------------------
// Routine Name : locateSinewave
// Input        : filename = .wav input filename
//                samplingRate : sampling rate of the audio data
//                freq = sinewave frequency to look for (10 contiguous periods)
// Return       : first sample position after sinewave or error condition
// Description  : locate the 10 periods sinewave preamble
//----------------------------------------------------------------------
static int locateSinewave(char *filename, int samplingRate, int freq) {
  int err, pos;
  long nsamples, countHalfPeriods, countSamples, totalCountSamples;
  float minLevel, sample;
  getsample *desc;

  pos = -1;
  for (;;) {
    err = RET_ERR;

    if ((desc = getsampleOpen(filename)) == NULL)
      break;

    minLevel = 0.05;
    countHalfPeriods = 0;
    nsamples = samplingRate/freq;
 
    // look for trigger mark (if trigger has been armed, PL will insert a trigger mark on first non-zero sample) 
    for (;;) {
      sample = getsampleRead(desc, TRUE);
      if (sample == -10)  // end-of-file
        break;
      if ((short)(sample * 32768.0f) == (short)0xa55a)  // trigger
        break;
    }
    if (sample == -10)  // end-of-file
      break;

    pos = desc->curpos;
    err = RET_OK;
    break;  // exit without looking for sinewave (horn tweeters problem)

    for (;;) {
      countSamples = getCountSamples(desc, minLevel);
      if (countSamples == 0)
        break;
      if (countSamples >= nsamples * 3 || countSamples < 0) {
        countHalfPeriods = 0;
        desc->crosspos = desc->curpos;
        desc->crossSample = desc->curSample;
      }
      else if (countSamples > 0) {
        if (countHalfPeriods == 0) {
          desc->crosspos = desc->curpos - 1;
          desc->crossSample = desc->curSample;
        }
        countHalfPeriods++;
        if (countHalfPeriods == 2*10) { // 10 periods
          countHalfPeriods = 0;
          totalCountSamples = desc->curpos - desc->crosspos;
          if ((totalCountSamples >= nsamples * 9) && (totalCountSamples <= nsamples * 11)) {
            pos = desc->crossSample >= 0 ? desc->crosspos - 1 : desc->crosspos + 1;
            pos += (nsamples * NB_SINEWAVES);
            if (desc->curpos < 200000)  // sinewave preamble
              err = RET_OK;
            break;
          }
        }
      }
    }
    break;  // exit the forever loop
  }
  if (desc != NULL)
    getsampleClose(desc);

  return(err == RET_OK ? pos : RET_ERR);
}

//----------------------------------------------------------------------
// Routine Name : getCountSamples
// Input        : desc = local descriptor pointer
//                minlevel = minimun level to look for
// Return       : . 0 if end of file
//                . > 0 if found (number of samples read)
//                . < 0 if not found (number of samples read)
// Description  : counts the number of samples before change in direction
//----------------------------------------------------------------------
static int getCountSamples(getsample *desc, float minlevel) {
  float min, max, cursample, nextsample;
  long dir, countSamples;

  countSamples = 1;
  min = 100.0;
  max = -100.0;
  dir = -1;
  cursample = desc->lastSample;
  countSamples++;
  nextsample = getsampleRead(desc, FALSE);
  for (;;) {

    if (cursample == -10 || nextsample == -10)
      return(0);	// done

    if (cursample > max) max = cursample;
    if (cursample < min) min = cursample;

    if (dir == -1)	// first time in loop
      dir = nextsample > cursample ? 1 : 0; // increasing or decreasing
      
    if (dir == 1) { // test if higher
      if (nextsample < cursample)
        break;
    }
    else { // test if lower
      if (nextsample > cursample)
        break;
    }
    cursample = getsampleRead(desc, TRUE);
    countSamples++;
    nextsample = getsampleRead(desc, FALSE);
  }
  if ((1.0+max) - (1.0+min) > minlevel/4) // check min amplitude
    return(countSamples);

  return(-countSamples);
}

//----------------------------------------------------------------------
// Routine Name : getsampleOpen
// Input        : filename = .wav input filename
// Return       : local descriptor pointer
// Description  : open the audio data file to locate the sinewave preamble position
//----------------------------------------------------------------------
static getsample *getsampleOpen(char *filename) {
  getsample *desc;

  desc = &gs_desc;
  desc->wave = new Wavefile();
  if (desc->wave->open(filename, FILE_READ) != RET_OK)
    return NULL;

  desc->curpos = 0;
  desc->crosspos = 0;
  desc->bufindex = 0;
  desc->bufsize = 0;
  desc->lastSample = 0;
  return(desc);
}

//----------------------------------------------------------------------
// Routine Name : getsampleRead
// Input        : desc = local descriptor pointer
//                inc = flag whether to increment the current sample position or not
// Return       : sample value
// Description  : read next sample
//----------------------------------------------------------------------
static float getsampleRead(getsample *desc, bool inc) {
  float sample;
  unsigned int br;

  if (desc->bufindex >= desc->bufsize) {

    if ((br = desc->wave->read(desc->buf, FRAMES_PER_BLOCK * SAMPLES_PER_FRAME)) == 0)
      return(-10);

    desc->bufsize = br;
    desc->bufindex = 0;
    if (desc->bufsize == 0)
      return(-10);	// end of file
  }
  sample = desc->buf[desc->bufindex];
  if (inc == TRUE) {
    desc->bufindex++;
    desc->curpos++;
    desc->lastSample = sample;
  }
  desc->curSample = sample;
  return(sample);
}

//----------------------------------------------------------------------
// Routine Name : getsampleClose
// Input        : desc = local descriptor pointer
// Return       : none
// Description  : close the audio data file
//----------------------------------------------------------------------
static void getsampleClose(getsample *desc) {
  delete desc->wave;
}
