/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : logsweep.cpp
 * This copy created : Apr 27, 2015
 * Version           : 1.10
 * Description       : generate logsweep audio data class (source)
 *                     originated from Angelo Farina
 ***************************************************************************/
#include <math.h>
#include "defs.h"
#include "logsweep.h"

#define PI     3.14159265358979323846
#define LOG2   0.69314718055994530941723212145818

//----------------------------------------------------------------------
// Routine Name : Logsweep
// Input        : none
// Return       : none
// Description  : initialization defaults (constructors)
//----------------------------------------------------------------------
Logsweep::Logsweep() {
  Type(LOGSWEEP);
  initialized = false;
  inverse = false;
  lowFrequency = LOW_FREQ;
  highFrequency = HIGH_FREQ;
  magnitude = 1.0;
}
Logsweep::Logsweep(SourceTrack track, int lowFrequency, int highFrequency, unsigned int totalSamples) {
  Type(LOGSWEEP);
  initialized = false;
  inverse = false;
  lowFrequency = LOW_FREQ;
  highFrequency = HIGH_FREQ;
  magnitude = 1.0;
  Track(track);
  this->lowFrequency = lowFrequency;
  this->highFrequency = highFrequency;
  TotalSamples(totalSamples);
}

//----------------------------------------------------------------------
void Logsweep::LowFrequency(float frequency) { this->lowFrequency = frequency; }
void Logsweep::HighFrequency(float frequency) { this->highFrequency = frequency; }
void Logsweep::Inverse(bool inverse) { this->inverse = inverse; }

//----------------------------------------------------------------------
// Routine Name : read
// Input        : bufp = buffer pointer where to read data
//                nSamples = number of samples to read
// Return       : number of data read
// Description  : generate and read a block of logsweep data
//----------------------------------------------------------------------
int Logsweep::read(float *bufp, unsigned int nSamples) {
  double ii = 0, s = 0, s2 = 0;
  unsigned int i = 0, count = 0;
  double om1 = 0, om2 = 0;
  int samplingRate;
  unsigned int totalSamples;
  unsigned int curSample;

  samplingRate = SamplingRate();
  totalSamples = TotalSamples();
  curSample = CurSample();

  if (!initialized) {
    om1 = 2.0*PI*lowFrequency/samplingRate;
    om2 = 2.0*PI*highFrequency/samplingRate;
    Noct = log(om2/om1) / log(2.0);
    N = totalSamples / Noct;
    om = pow(2.0,(-Noct))*PI*N/LOG2*highFrequency/samplingRate*2.0;
    initialized = true;
  }
  count = 0;
  if (Track() != MONO)
    nSamples /= 2; // stereo

  if (nSamples > totalSamples - curSample)
    nSamples = totalSamples - curSample;

  for (i = curSample; i < curSample+nSamples; i++) {
    ii = (inverse ?(double)(totalSamples-i-1) : (double)i);
    s = magnitude * sin(om*(pow(2.0,ii/N)-1.0));
    if (inverse)  // inverse logsweep
      s = s*pow(2.0,(-Noct*(totalSamples-ii-1.0)/totalSamples));

    switch (Track()) {
      case MONO:
        *bufp++ = (float)s; 
        break;
      case STEREO_LEFT:
        *bufp++ = (float)s; 
        *bufp++ = 0;
        break;
      case STEREO_RIGHT:
        *bufp++ = 0;        
        *bufp++ = (float)s;
        break;
      case STEREO_BOTH:
        *bufp++ = (float)s;
        *bufp++ = (float)s;
        break;
    }
    count++;
  }
  curSample += count;
  if (Track() != MONO)
    count *= 2;

  CurSample(curSample);
  return(count);
}

unsigned long Logsweep::seek() {
  CurSample(Pos());
  return(CurSample());
}


