/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : main.c
 * This copy created : May 16, 2015
 * Version           : 1.61
 * Description       : audio FreeRTOS task
 *                     allows:
 *                       . playback of standard CD wavefile with room correction and crossovers
 *                       . playback from CD player with room correction and crossovers
 *                       . measurement of logsweep and conversion to IR
 *                       . measurement of speakers propagation delay
 ***************************************************************************/
 #include <__cross_studio_io.h>

#include <limits.h>
#include <math.h>
#include <string.h>

#include "FreeRTOS.h"
#include "task.h"
#include "semphr.h"

#include "defs.h"
#include "xparameters.h"  // Zynq parameters
#include "Zynq7000_intc.h"
#include "cache.h"
#include "fifo.h"
#include "cdma.h"
#include "convolver.h"
#include "xover.h"
#include "fatfs.h"
#include "wdt.h"
#include "play.h"
#include"ir.h"

ISR_FN_t interrupt_handlers[160];

SemaphoreHandle_t flippedSemaphore; // CDMA OUT transfer completed
int exitPlay;

static FATFS FatFs;  // FatFs work area needed for each volume

static float convolverAdjust;  // attenuation to avoid clipping at the convolution stage

static float level[NB_SET];  // volume applied to each relevant sample (in dB)

int activeBufout;  // switch between bufout0 and bufout1
short *blankBuf = (short *)BUF_OUT;
short *shortBufout0 = (short *)BUF_OUT0;
short *shortBufout1 = (short *)BUF_OUT1;

static float *floatBuf = (float *)BUF_FLOAT;
static float *floatBufXO = (float *)BUF_FLOATXO;
short *recordBuf;  // record buffer spdif_in

// delays
#define MAX_DELAY 20  // in millisecond
#define DELAY_BUFSIZE  ((SAMPLING_RATE * MAX_DELAY)/1000)*2
static float delayBuf0[DELAY_BUFSIZE], delayBuf1[DELAY_BUFSIZE];
static unsigned int delayMax;  // highest set delay
static unsigned int delay[NB_SET];  // values returned during calibration
static unsigned int delayIndex[NB_SET];  // current index position

// crossovers
static int cutoff1 = 0;
static int filterOrder1 = 2;  // default 12dB/octave
static int cutoff2 = 0;
static int filterOrder2 = 2;  // default 12dB/octaveS

static unsigned int nbChan = SAMPLES_PER_FRAME; // number of output channels (default = 2)

static void paramsInit();
static void initLevel();
static void initDelay();
static void clearBuffers();
static void xover(float *src, float *dest, int nb_chan, int nSamples);
static void copyDelayBuf(float *src, float *dest, unsigned int len, unsigned int bufSize);
static float getDelayedValue(int idx);
static void float_to_short(float *sourceBuffer, short *targetBuffer, int nSamples);

// ---------------------------------------------------------------------------
void vApplicationMallocFailedHook(void);
void vApplicationIdleHook(void);
void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName);
void vApplicationTickHook(void);

static void prvSetupHardware(void);

#define main_TASK_PRIORITY (tskIDLE_PRIORITY + 1)
static void audioTask(void *pvParameters);  // task

//----------------------------------------------------------------------
// Routine Name : copyDelayBuf
// Input        : src = audio data source pointer
//                dest = delay buffer destination pointer
//                len = max size
//                bufSize = audio data buffer size
// Return       : none
// Description  : copy data chunck corresponding to the delay shift (from end of data buffer)
//----------------------------------------------------------------------
static void copyDelayBuf(float *src, float *dest, unsigned int len, unsigned int bufSize) {
  float *inbufp;

  if (len != 0) {
    if (bufSize > len) {
      inbufp = src + (bufSize - len);
      for (int i = 0; i < len; i++)
        *dest++ = *inbufp++;
    }
    else
      for (int i = 0; i < len; i++)
        *dest++ = 0;
  }
}

//----------------------------------------------------------------------
// Routine Name : getDelayedValue
// Input        : idx = index of data
// Return       : sample value
// Description  : get the next audio sample (from delay buffer or normal buffer)
//----------------------------------------------------------------------
static float getDelayedValue(int idx) {
  float *inbufp;

  if (delayIndex[idx] < delayMax)
    inbufp = (activeBufout == 0 ? delayBuf0 : delayBuf1) + delayIndex[idx];
  else
    inbufp = floatBuf + (delayIndex[idx]-delayMax);
  
  if (idx % 2 != 0)
    inbufp++; // right channel
  delayIndex[idx] += 2;

  return(*inbufp);
}

//----------------------------------------------------------------------
// Routine Name : float_to_short
// Input        : src = normalized audio data source pointer
//                dest = 16bit audio data destination pointer
//                nSamples = number of samples to convert
// Return       : none
// Description  : convert bufer of float to short
//----------------------------------------------------------------------
static void float_to_short(float *src, short *dest, int nSamples) {
  for (int i = 0; i < nSamples; i++) {
    short samp = (short) (*src * (32767.0f));
    *dest = samp;
    src++;
    dest++;
  }
}

//----------------------------------------------------------------------
// Routine Name : initLevel
// Input        : none
// Return       : none
// Description  : volume levels initialization
//----------------------------------------------------------------------
static void initLevel() {
  int volume, balanceVal, balanceLeft, balanceRight;
  int lev[NB_SET];
 
  volume = 0;
  balanceVal = 0;
  lev[FPL] = 0;
  lev[FPR] = 0;
  lev[LPL] = 0;
  lev[LPR] = 0;
  lev[BPL] = 0;
  lev[BPR] = 0;
  lev[HPL] = 0;
  lev[HPR] = 0;
 
  balanceLeft = (balanceVal < 0 ? balanceVal: 0);
  balanceRight = (balanceVal > 0 ? -balanceVal: 0);

  for (int i = 0; i < NB_SET; i += 2) {
    level[i] = (float)pow(10.0, (float)(volume + balanceLeft + lev[i]) / 20.0);
    level[i+1] = (float)pow(10.0, (float)(volume + balanceRight + lev[i+1]) / 20.0);
  }
}

//----------------------------------------------------------------------
// Routine Name : initDelay
// Input        : none
// Return       : none
// Description  : speaker delays initialization
//----------------------------------------------------------------------
static void initDelay() {
  // values set during calibration (latency + sound travel time)
  delay[FPL] = 0;
  delay[FPR] = 0;
  delay[LPL] = 0;
  delay[LPR] = 0;
  delay[BPL] = 0;
  delay[BPR] = 0;
  delay[HPL] = 0;
  delay[HPR] = 0;

  // stereo signal so copy 2 times the number of samples to buffer
  for (int i = 0; i < NB_SET; i++)
    delay[i] *= 2;

  // find the highest value
  delayMax = 0;
  for (int i = 0; i < NB_SET; i++)
    if (delayMax < delay[i])
      delayMax = delay[i];
}

//----------------------------------------------------------------------
// Routine Name : paramsInit
// Input        : none
// Return       : none
// Description  : general parameters initialization
//----------------------------------------------------------------------
static void paramsInit() {
  exitPlay = FALSE;
  activeBufout = 0;

  // filter attenuation to avoid clipping during convolution (0dB -> 300.0)
  convolverAdjust = 300.0;  // formula = 300 + (x * 10) example: for -4.2dB set convolverAdjust to 258.0
  convolverAdjust = (float)pow(10.0, ((float)convolverAdjust - (float)300.0) / 200.0);

  cutoff1 = 200;  // 200 Hz LP/HP
  cutoff2 = 0;
  filterOrder1 = 4;
  filterOrder2 = 0;

  if (cutoff1)
    nbChan += 2;
  if (cutoff2)
    nbChan += 2;
	
  initLevel();
  initDelay();

  recordBuf = (short *)BUF_RECORD;  // recorded data
}

//----------------------------------------------------------------------
// Routine Name : clearBuffers
// Input        : none
// Return       : none
// Description  : clear all residual data
//----------------------------------------------------------------------
static void clearBuffers() {

  // clear the 2 FPGA bram blocks
  CDMA_transfer(1, (UINT32)blankBuf, XPAR_BRAM_0_BASEADDR, BRAM_OUT__DEPTH*(BRAM_WIDTH/8), FALSE);
  CDMA_transfer(1, (UINT32)blankBuf, XPAR_BRAM_1_BASEADDR, BRAM_OUT__DEPTH*(BRAM_WIDTH/8), FALSE);

  // clear playback buffers
  for (int i = 0; i < NB_FRAMES*nbChan; i++) {
    shortBufout0[i] = 0;
    shortBufout1[i] = 0;
  }
  // clear delay buffers
  for (int i = 0; i < DELAY_BUFSIZE; i++) {
    delayBuf0[i] = 0;
    delayBuf1[i] = 0;
  }
  // clear convolver buffers
  convolverInit(NB_FRAMES);

  xoverClear();
}

//----------------------------------------------------------------------
// Routine Name : xover
// Input        : src = audio data source pointer
//                dest = audio data destination pointer
//                nb_chan = number of crossovers
//                nSamples = number of stereo samples  
// Return       : 
// Description  : apply crossover to buffer
//----------------------------------------------------------------------
static void xover(float *src, float *dest, int nb_chan, int nSamples) {
  float sampleLeft, sampleRight;

  for (int i = 0; i < nSamples; i++) {
    sampleLeft = *src++;
    sampleRight = *src++;

    if (nb_chan == 2) {
      if (delay[FPL] != 0)
        sampleLeft = getDelayedValue(FPL);
      if (delay[FPR] != 0)
        sampleRight = getDelayedValue(FPR);
  
      *dest++ = sampleLeft * level[FPL];
      *dest++ = sampleRight * level[FPR];
    }
    if (nb_chan >= 4) {
      if (delay[HPL] != 0)
        sampleLeft = getDelayedValue(HPL);
      if (delay[HPR] != 0)
        sampleRight = getDelayedValue(HPR);
#ifdef XOVER
      *dest++ = highpass(LEFT_TRACK, sampleLeft) * level[HPL];
      *dest++ = highpass(RIGHT_TRACK, sampleRight) * level[HPR];
#else
      *dest++ = sampleLeft * level[HPL];
      *dest++ = sampleRight * level[HPR];
#endif
    }
    if (nb_chan >= 6) {
      if (delay[BPL] != 0)
        sampleLeft = getDelayedValue(BPL);
      if (delay[BPR] != 0)
        sampleRight = getDelayedValue(BPR);
#ifdef XOVER
      *dest++ = bandpass(LEFT_TRACK, sampleLeft) * level[BPL];
      *dest++ = bandpass(RIGHT_TRACK, sampleRight) * level[BPR];
#else
      *dest++ = sampleLeft * level[BPL];
      *dest++ = sampleRight * level[BPR];
#endif
    }
    if (nb_chan > 2) {
      if (delay[LPL] != 0)
        sampleLeft = getDelayedValue(LPL);
      if (delay[LPR] != 0)
        sampleRight = getDelayedValue(LPR);
#ifdef XOVER
      *dest++ = lowpass(LEFT_TRACK, sampleLeft) * level[LPL];
      *dest++ = lowpass(RIGHT_TRACK, sampleRight) * level[LPR];
#else
      *dest++ = sampleLeft * level[LPL];
      *dest++ = sampleRight * level[LPR];
#endif
    }
  }
}

//----------------------------------------------------------------------
// Routine Name : main
// Input        : none
// Return       : none
// Description  : FreeRTOS main task
//----------------------------------------------------------------------
int main(void) {
  prvSetupHardware();
  xTaskCreate(audioTask, (signed char *)"HW", 4096, NULL, main_TASK_PRIORITY, NULL);
  vTaskStartScheduler();
  for(;;)
    ;  // never returns
}

//----------------------------------------------------------------------
// Routine Name : audioTask
// Input        : pvParameters = FreeRTOS parameters (not used)
// Return       : none
// Description  : audio task application
//----------------------------------------------------------------------
static void audioTask(void *pvParameters) {
  UINT sRead;
  short *shortBuf;
  char filename[16];
  int err;

  // user parameters
  // ---------------
  SourceType type = CDPLAYER;   // WAVEFILE/CDPLAYER/LOGSWEEP_SIN
  SourceTrack track = STEREO_LEFT;  // STEREO_LEFT/STEREO_RIGHT (LOGSWEEP_SIN only)
  int bypassMeasure = FALSE;        // if TRUE do LOGSWEEP_SIN IR conversion only (LOGSWEEP_SIN only and for testing)

  for (;;) {
    err = RET_ERR;
    paramsInit();

    if (xoverInit(SAMPLING_RATE, cutoff1, cutoff2, filterOrder1, filterOrder2) != RET_OK)
      break;

    for (int i = 0; i < NB_FRAMES*nbChan; i++)
      blankBuf[i] = 0;
    clearBuffers();

    Xil_DCacheEnable();
    
    if (f_mount(&FatFs, "", 0) != FR_OK)
      break;
    
    switch (type) {
    case WAVEFILE:
    case CDPLAYER:
      // get convolver ready
      convolverInit(NB_FRAMES);
      convolverScaleCoef2(convolverAdjust);
#ifdef DRC
      if (convolverImpulse(WAVEFILE, "drc.wav") != RET_OK)
        break;
#endif
      if (type == WAVEFILE)
        strcpy(filename, "music/bach2.wav");
      else
        strcpy(filename, "");
      play_open(type, filename);
      break;

    // signal generator (mic PGA=0dB)
    case LOGSWEEP_SIN:
      //if (bypassMeasure == FALSE)
      //  vTaskDelay(10000 / portTICK_PERIOD_MS);  // 10s delay to get out of the way

      strcpy(filename, "ms");  // measurement
      strcat(filename, (track == STEREO_LEFT ? "Left" : "Right"));
      strcat(filename, ".wav");
      if (bypassMeasure == TRUE)
        err = ir(filename, LOGSWEEP_SAMPLES); // do the IR here (when operations split in two: measurement/ir)
      else {
        play_open(type, filename);  // record data to filename
        play_setTrack(track);  // NOTE: in loopback mode only SPDIF LEFT channel is read (mic is mono)
      }
      break;
    }
    if ((type == LOGSWEEP_SIN) && (bypassMeasure == TRUE))
      break;  // no measurement -> exit the forever loop

    int firstLoop = TRUE;
    int done = FALSE;
    for (;;) {	// playback loop
      // cdma writes from shortbuf0 when activeBuf = 0
      // -> when activeBuf = 0 then copy output data (SD or signalGen) to shortbuf1
      shortBuf = activeBufout == 0 ? shortBufout1 : shortBufout0;

      if ((sRead = play_read(floatBuf, NB_FRAMES * SAMPLES_PER_FRAME)) == 0)
        done = TRUE;
      else {
#ifdef DRC
        if ((type == WAVEFILE) || (type == CDPLAYER))
          convolve(floatBuf);  // no convolution when doing a calibration (LOGSWEEP_SIN)
#endif
        // copy delay data chunk into opposite delay buffer (i.e. accessed first after next flip)
        copyDelayBuf(floatBuf, activeBufout == 0 ? delayBuf1 : delayBuf0, delayMax, NB_FRAMES*SAMPLES_PER_FRAME);
        for (int i = 0; i < NB_SET; i++)
          delayIndex[i] = delayMax - delay[i];

        xover(floatBuf, floatBufXO, nbChan, NB_FRAMES);
        float_to_short(floatBufXO, (short *)shortBuf, NB_FRAMES*nbChan);

        if (firstLoop == TRUE) {
          firstLoop = FALSE;
          // enable FIFOs interrupt
          FIFO_REG(FIFO0)->IER |= XLLF_INT_ALL_MASK;
          FIFO_REG(FIFO1)->IER |= XLLF_INT_ALL_MASK;
        }
      }
      xSemaphoreTake(flippedSemaphore, portMAX_DELAY);  // wait for CDMA completion before proceeding to next buffer
      if (done != FALSE)
        break;  // exit the forever loop
    }
    // disable FIFOs interrupt
    FIFO_REG(FIFO0)->IER &= ~XLLF_INT_ALL_MASK;  
    FIFO_REG(FIFO1)->IER &= ~XLLF_INT_ALL_MASK;
    vTaskDelay(100);

    // save the recorded data if measurement
    if (type == LOGSWEEP_SIN)
      play_save((short *)BUF_RECORD, (recordBuf - (short *)BUF_RECORD) * sizeof(short)); // write recorded data

    play_close();

    //if (type == LOGSWEEP_SIN)
    //  if (ir(filename, LOGSWEEP_SAMPLES) != RET_OK)
    //    break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  debug_printf("audioTask status = %d\n", err);
  for (;;)
    vTaskDelay(100);   // never return from the task
}

//----------------------------------------------------------------------
// Routine Name : prvSetupHardware
// Input        : none
// Return       : none
// Description  : system initialization
//----------------------------------------------------------------------
static void prvSetupHardware(void) {
  // Ensure no interrupts execute while the scheduler is in an inconsistent state.
  // Interrupts are automatically enabled when the scheduler is started
  portDISABLE_INTERRUPTS();

  CPU_INTC_Initialize(XPAR_CPU_ID + 1); // done by cpu0
  
  if (CDMA_init(CDMA0) != RET_OK)
    for (;;)
      ;
  if (FifoInit(FIFO0) != RET_OK)
    for (;;)
      ;
  if (FifoInit(FIFO1) != RET_OK)
    for (;;)
      ;    
  if ((flippedSemaphore = xSemaphoreCreateBinary()) == NULL)
    for (;;)
      ;  
}

//-----------------------------------------------------------
void vApplicationMallocFailedHook(void) {
  // Called if a call to pvPortMalloc() fails because there is insufficient
  // free memory available in the FreeRTOS heap.  pvPortMalloc() is called
  // internally by FreeRTOS API functions that create tasks, queues, software
  // timers, and semaphores.  The size of the FreeRTOS heap is set by the
  // configTOTAL_HEAP_SIZE configuration constant in FreeRTOSConfig.h
  taskDISABLE_INTERRUPTS();
  for(;;)
    ;
}

//-----------------------------------------------------------
void vApplicationStackOverflowHook(TaskHandle_t pxTask, char *pcTaskName) {
  (void) pcTaskName;
  (void) pxTask;

  // Run time stack overflow checking is performed if
  // configCHECK_FOR_STACK_OVERFLOW is defined to 1 or 2
  for(;;)
    ;
}

//-----------------------------------------------------------
void vApplicationIdleHook(void) {
  volatile size_t xFreeHeapSpace;

  // This is just a trivial example of an idle hook.  It is called on each
  // cycle of the idle task.  It must *NOT* attempt to block.  In this case the
  // idle task just queries the amount of FreeRTOS heap that remains.  See the
  // memory management section on the http://www.FreeRTOS.org web site for memory
  // management options.  If there is a lot of heap memory free then the
  // configTOTAL_HEAP_SIZE value in FreeRTOSConfig.h can be reduced to free up RAM
  xFreeHeapSpace = xPortGetFreeHeapSize();

  // Remove compiler warning about xFreeHeapSpace being set but never used
  (void)xFreeHeapSpace;
}

//-----------------------------------------------------------
void vAssertCalled(const char *pcFile, unsigned long ulLine) {
  volatile unsigned long ul = 0;
  (void) pcFile;
  (void) ulLine;

  taskENTER_CRITICAL();
  {
    // Set ul to a non-zero value using the debugger to step out of this function
    while(ul == 0) {
      portNOP();
    }
  }
  taskEXIT_CRITICAL();
}

//-----------------------------------------------------------
void vApplicationTickHook(void) { }

//---------------------------------------------------------
void vInitialiseTimerForRunTimeStats(void) {
  uint32_t ulValue;
  const uint32_t ulMaxDivisor = 0xff, ulDivisorShift = 0x08;

  ulValue = WDT->CONTROL;
  ulValue |= ulMaxDivisor << ulDivisorShift;
  WDT->CONTROL = ulValue;
  
  WDT->LOAD = UINT_MAX;
  WDT->DISABLE = XSCUWDT_DISABLE_VALUE1;
  WDT->DISABLE = XSCUWDT_DISABLE_VALUE2;
  WDT->CONTROL |= XSCUWDT_CONTROL_WD_ENABLE_MASK;
}
