/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : play.cpp
 * This copy created : May 16, 2015
 * Version           : 1.36
 * Description       : source front-end (C -> C++)
 ***************************************************************************/
#include <vector>

#include "defs.h"
#include "xparameters.h"  // Zynq parameters

#include "wavefile.h"
#include "cdp.h"
#include "logsweep.h"
#include "sinewave.h"
#include "blank.h"
#include "trigger.h"
#include "measure.h"
#include "play.h"

typedef enum MeasureSequence {
  M_NULL = 0,        // end of sequence
  M_HEADER = 1,      // header of blanks
  M_TRAILER = 2,     // trailer of blanks (should be at least 2 times shortBufout size)
  M_SINEWAVE = 3,    // sinewave signal (10 periods preamble during testing with no crossover)
  M_LOGSWEEP = 4,    // logsweep signal
  M_LEFT = 5,        // stereo left track
  M_RIGHT = 6,       // stereo right track
  M_ARM_TRIGGER_NZERO = 7,  // arm the FPGA trigger so we know exactly the beginning/end of relevant data (system latency)
  M_ARM_TRIGGER_ZERO = 8  // same but for testing (no crossover)
} MeasureSequence;

// audio generator sequences for measurements
static MeasureSequence msLogsweep_sin[] = { M_HEADER, M_ARM_TRIGGER_NZERO, M_LOGSWEEP, M_TRAILER, M_NULL };

static Source *source = NULL;
static Wavefile *record = NULL;

// FPGA AXI GPIO
typedef struct {
  UINT32 Chan1Data;
  UINT32 Chan1Tri;
  UINT32 Chan2Data;
  UINT32 Chan2Tri;
  UINT32 nc[43];
  UINT32 Gier;
  UINT32 Ier;
  UINT32 Isr;
} GPIO_Type;
#define GPIO ((volatile GPIO_Type*)(XPAR_SPDIF_AXI_GPIO_0_BASEADDR))

// gpio bit0: pause/play
#define PAUSE_PLAY 0x01

// gpio bit1: mute
#define MUTE 0x02

// gpio bit2: spdif_in enable/disable (calibration)
#define RECORD 0x04

// gpio bit3: arm/reset trigger
// when the trigger is armed from PS, the first non-zero value outputted to spdif 
// will replace the corresponding spdif input sample value with 0xa55a
// the trigger will have to be reseted from PS before a new arming
#define TRIGGER 0x08

// gpio bit4: trigger polarity
#define TRIGGER_POL 0x10

// gpio bit5: trigger track left/right
#define TRIGGER_TRACK 0x20

// gpio bit6: mono/stereo recording
#define RECORD_MONO 0x40

static unsigned int gpio_control = 0;  // copy because fpga ChanxData is write-only

static int parse(MeasureSequence *ms);

//----------------------------------------------------------------------
// Routine Name : play_open
// Input        : type = 
//                filename = 
// Return       : error condition
// Description  : initialization and invoques virtual open function (source class)
//----------------------------------------------------------------------
int play_open(SourceType type, char *filename) {
  int err;

  err = RET_ERR;
  switch (type) {
  case WAVEFILE:
    if ((source = new Wavefile()) == NULL)
      break;
    if (source->open(filename, FILE_READ) != RET_OK)
      break;
    play_setRecord(FALSE);  // disable FPGA spdif_in
    err = RET_OK;
    break;

  case CDPLAYER:
    if ((source = new Cdp()) == NULL)
      break;
    play_setMono(FALSE);  // stereo mode
    play_setRecord(TRUE);  // enable FPGA spdif_in
    err = RET_OK;
    break;

  case LOGSWEEP_SIN:
    if ((source = new Measure(type)) == NULL)
      break;
    if (parse(msLogsweep_sin) != RET_OK)
      break;
    record = new Wavefile();
    if (record->open(filename, FILE_WRITE) != RET_OK)
      break;
    play_setMono(TRUE);
    play_setRecord(TRUE);  // enable FPGA spdif_in
    err = RET_OK;
    break;

  case INVLOGSWEEP:
    if ((source = new Logsweep()) == NULL)
      break;
    source->Type(INVLOGSWEEP);
    source->Inverse(true);
    source->Track(MONO);
    source->TotalSamples(LOGSWEEP_SAMPLES);
    err = RET_OK;
    play_setRecord(FALSE);  // disable FPGA spdif_in
    break;

  default:
    break;
  }
  return err;
}

//----------------------------------------------------------------------
int play_getTrack() { return source->Track(); }
void play_setTrack(SourceTrack track) { source->Track(track); }

// sinewave setting
void play_setFrequency(unsigned int frequency) {
  if (source->Type() == LOGSWEEP_SIN) {
    std::vector<Source*>p = ((Measure *)source)->getVector();
    for (std::vector<Source*>::iterator i = p.begin(); i != p.end(); ++i) {
      if ((*i)->Type() == SINEWAVE) {
        (*i)->Frequency(frequency);
        (*i)->TotalSamples(source->SamplingRate()/frequency*NB_SINEWAVES);  // 10 periods
      }
    }
  }
}

//----------------------------------------------------------------------
// Routine Name : play_seek
// Input        : pos = new file position
// Return       : none
// Description  : virtual seek function (source class)
//----------------------------------------------------------------------
void play_seek(unsigned long pos) {
  source->Pos(pos);
  source->seek();
}

//----------------------------------------------------------------------
// Routine Name : play_read
// Input        : bufp = destination pointer
//                nSamples = number of samples to read
// Return       : number of samples read
// Description  : virtual read function (source class)
//----------------------------------------------------------------------
int play_read(float *bufp, int nSamples) {
  UINT sRead = source->read(bufp, nSamples);
  if (sRead < nSamples && sRead != 0) {
    // fill the remainder with blanks
    int count = nSamples - sRead;
    bufp += sRead;
    for (int i = 0; i < count; i++)
      *bufp++ = 0.0;
    sRead = nSamples;
  }
  return sRead;
}

//----------------------------------------------------------------------
// Routine Name : play_save
// Input        : bufp = src pointer
//                nSamples = number of samples to write
// Return       : number of written samples or error condition
// Description  : virtual write function (source class)
//----------------------------------------------------------------------
int play_save(short *bufp, int nSamples) {
  return(record->write(bufp, nSamples));
}

//----------------------------------------------------------------------
// Routine Name : play_close
// Input        : none
// Return       : none
// Description  : terminates the source classes
//----------------------------------------------------------------------
void play_close() {
  play_setRecord(FALSE);  // disable FPGA spdif_in
  play_setTrigger(FALSE);  // reset FPGA trigger

  if (source != NULL) { delete source; source = NULL; }
  if (record != NULL) { delete record; record = NULL; }
}

//----------------------------------------------------------------------
// Routine Name : play_setPause
// Input        : pause = enable/disable value
// Return       : none
// Description  : control FPGA play/pause 
//----------------------------------------------------------------------
void play_setPause(int pause) {
  if (pause == FALSE)
    gpio_control &= ~PAUSE_PLAY;  // play
  else
    gpio_control |= PAUSE_PLAY;  // pause

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : play_setMute
// Input        : mute = enable/disable value
// Return       : none
// Description  : control FPGA mute
//----------------------------------------------------------------------
void play_setMute(int mute) {
  if (mute == FALSE)
    gpio_control &= ~MUTE;
  else
    gpio_control |= MUTE;

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : play_setRecord
// Input        : enable = enable/disable value
// Return       : none
// Description  : allows the FPGA to write to the spdif_in FIFO or not
//----------------------------------------------------------------------
void play_setRecord(int enable) {
  if (enable == FALSE)
    gpio_control &= ~RECORD;  // disable FPGA spdif_in
  else
    gpio_control |= RECORD;  // enable FPGA spdif_in

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : play_setTrigger
// Input        : arm = enable/disable value
// Return       : none
// Description  : reset/arm the FPGA trigger
//                (must be reseted before arming)
//----------------------------------------------------------------------
void play_setTrigger(int arm) {
  if (arm == FALSE)
    gpio_control &= ~TRIGGER;  // reset FPGA trigger
  else
    gpio_control |= TRIGGER;  // arm FPGA trigger

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : play_setTriggerPol
// Input        : pol = enable/disable value
// Return       : none
// Description  : set the FPGA trigger polarity
//                FALSE -> trigger on the first 3 zero samples
//                TRUE -> trigger on the first non-zero sample
//----------------------------------------------------------------------
void play_setTriggerPol(int pol) {
  if (pol == FALSE)
    gpio_control &= ~TRIGGER_POL;
  else
    gpio_control |= TRIGGER_POL;

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : play_setTriggerTrack
// Input        : track = left/right (1/0)
// Return       : none
// Description  : set the FPGA trigger polarity
//----------------------------------------------------------------------
void play_setTriggerTrack(int track) {
  if (track == RIGHT_TRACK)
    gpio_control &= ~TRIGGER_TRACK;  // right (spdif channelA = 0)
  else
    gpio_control |= TRIGGER_TRACK;  // left (spdif channelA = 1)

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : play_setMono
// Input        : mono = true for mono
// Return       : none
// Description  : set the FPGA mono/stereo for recording
//----------------------------------------------------------------------
void play_setMono(int mono) {
  if (mono == FALSE)
    gpio_control &= ~RECORD_MONO;  // spdif_in stereo
  else
    gpio_control |= RECORD_MONO;  // spdif_in mono

  GPIO->Chan1Data = gpio_control;
}

//----------------------------------------------------------------------
// Routine Name : parse
// Input        : meas = pointer to Meas array to parse
// Return       : error condition
// Description  : create the array of classes to generate the selected
//                audio data pattern
//----------------------------------------------------------------------
#define MAX_PATTERN 20

static int parse(MeasureSequence *ms) {
  int err;
  int count = 0;
  bool done = false;
  SourceTrack track = source->Track();

  for (;;) {
    err = RET_ERR;

    if (count++ > MAX_PATTERN)  // place a limit just in case (normally a sequence should finish with M_NULL)
      break;

    switch (*ms++) {
    case M_HEADER:
      ((Measure *)source)->getVector().push_back(new Blank(track, HEADER_SIZE));
      break;

    case M_SINEWAVE:
      ((Measure *)source)->getVector().push_back(new Sinewave(track, SINEWAVE_FP_FREQ, NB_SINEWAVES));
      break;

    case M_LOGSWEEP:
      ((Measure *)source)->getVector().push_back(new Logsweep(track, LOW_FREQ, HIGH_FREQ, LOGSWEEP_SAMPLES));
      break;

    case M_TRAILER:
      ((Measure *)source)->getVector().push_back(new Blank(track, TRAILER_SIZE));
      break;

    case M_ARM_TRIGGER_NZERO:  // trigger on the first non-zero sample
      ((Measure *)source)->getVector().push_back(new Trigger(track, TRUE));
      break;

    case M_ARM_TRIGGER_ZERO:  // trigger on the first 3 zero samples
      ((Measure *)source)->getVector().push_back(new Trigger(track, FALSE));
      break;

    case M_LEFT:
      track = STEREO_LEFT;
      break;

    case M_RIGHT:
      track = STEREO_RIGHT;
      break;

    case M_NULL:
      err = RET_OK;
    default:
      done = true;
      break;
    }
    if (done)
      break;
  }
  return err;
}
