/*****************************************************************************
* This file is the glue layer between file system and *driver.
* Description related to SD driver:
* Process to use file system with SD
* Select xilffs in SDK when creating a BSP
* In SDK, set "fs_interface" to 1 to select SD interface.
* This glue layer can currently be used only with one SD controller enabled.
* In order to use eMMC, in SDK set "Enable MMC" to 1. If not, SD support is enabled by default
*
* Description:
* This glue layer initializes the host controller and SD card in disk_initialize.
* If SD card supports it, 4-bit mode and high speed mode will be enabled.
* The default block size is 512 bytes.
* disk_read and disk_write functions are used to read and write files using ADMA2 in polled mode.
* The file system can be used to read from and write to an SD card that is already formatted as FATFS
******************************************************************************/
#include "sdio.h"
#include "defs.h"
#include "xparameters.h"
#include "xsdps.h"

#define HIGH_SPEED_SUPPORT	        0x01
#define WIDTH_4_BIT_SUPPORT	        0x4
#define SD_CLK_25_MHZ		        25000000

/****************************************************************************
* Gets the status of the disk. In case of SD, it checks whether card is present or not
* param	drv - Drive number
* return
*		0		Status ok
*		STA_NOINIT	Drive not initialized
*		STA_NODISK	No medium in the drive
*		STA_PROTECT	Write protected
* note		If Card detect signal is not connected,
*		this function will not be able to check if card is present.
******************************************************************************/
DSTATUS disk_status(BYTE drv) {
  DSTATUS status = 0;
  UINT32 statusReg;
  volatile SD_Type *sd = SD_REG(SD0);
 
  return RET_OK;
/*
  statusReg = sd->PRES_STATE;
  if ((statusReg & XSDPS_PSR_CARD_INSRT_MASK) == 0) {
     status = STA_NODISK | STA_NOINIT;
  }
  else {
    if ((statusReg & XSDPS_PSR_WPS_PL_MASK) == 0){
      status = STA_PROTECT;
    }
  }
  return status;*/
}

/****************************************************************************
* Initializes the drive.
* In case of SD, it initializes the host controller and the card.
* This function also selects additional settings such as bus width,
* speed and block size.
* param  drv - Drive number
* return  s - which contains an OR of the following information
*    STA_NODISK  Disk is not present
*    STA_NOINIT  Drive not initialized
*    STA_PROTECT  Drive is write protected
*    0 or only STA_PROTECT both indicate successful initialization.
******************************************************************************/
DSTATUS disk_initialize(BYTE drv) {
  DSTATUS status;
  int err;
  UINT8 SCR[8] __attribute__ ((aligned(32)));
  UINT8 rbuf[64] __attribute__ ((aligned(32)));

  for (;;) {
    if ((status = disk_status(drv)) & STA_NODISK) // Check if card is in the socket
      break;

    if (xsdpsIsInitialized(SD0) == FALSE) { // check if already initialized

      status = STA_NOINIT;
      if ((err = xsdpsInitialize(SD0)) != RET_OK)
        break;

      if ((err = xsdpsSdCardInitialize(SD0)) != RET_OK)
        break;

      if ((err = xsdpsChangeClockFreq(SD0, SD_CLK_25_MHZ)) != RET_OK)
        break;

      if ((err = xsdpsSelectCard(SD0)) != RET_OK)
        break;

      if ((err = xsdpsGetBusWidth(SD0, SCR)) != RET_OK)
        break;

      if ((err = xsdpsGetBusSpeed(SD0, rbuf)) != RET_OK)
        break;

      if (rbuf[13] & HIGH_SPEED_SUPPORT){
        if ((err = xsdpsChangeBusSpeed(SD0)) != RET_OK)
	  break;
      }
      if ((err = xsdpsPullup(SD0)) != RET_OK)
        break;

      if (SCR[1] & WIDTH_4_BIT_SUPPORT) {
        if ((err = xsdpsChangeBusWidth(SD0)) != RET_OK)
	  break;
      }
      if ((err = xsdpsSetBlockSize(SD0, XSDPS_BLK_SIZE_512_MASK)) != RET_OK)
        break;
    }
    // Disk is initialized
    status &= (~STA_NOINIT);
    break;  // exit the forever loop
  }
  return status;
}

/****************************************************************************
* Reads the drive
* In case of SD, it reads the SD card using ADMA2 in polled mode.
* @param  drv - Drive number
* @param  *buff - Pointer to the data buffer to store read data
* @param  sector - Start sector number
* @param  count - Sector count
* @return
*    RES_OK    Read successful
*    STA_NOINIT  Drive not initialized
*    RES_ERROR  Read not successful
******************************************************************************/
DRESULT disk_read(BYTE drv, BYTE *bufp, DWORD sector, BYTE count) {
  DSTATUS status = RES_OK;
  volatile SD_Params *sdpar = SD_PAR(SD0);

  for (;;) {
    status = RES_PARERR;
    if (!count)
      break;

    status = RES_NOTRDY;
    if (disk_status(drv) & STA_NOINIT)
      break;

    // Convert LBA to byte address if needed
    if (!(sdpar->HCS))
      sector *= XSDPS_BLK_SIZE_512_MASK;

    status = RES_ERROR;
    if ((xsdpsReadPolled(SD0, sector, count, bufp)) != RET_OK)
      break;

    status = RES_OK;
    break;  // exit the forever loop
  }
  return status;
}

/****************************************************************************
* Reads the drive
* In case of SD, it reads the SD card using ADMA2 in polled mode.
* param  drv - Drive number
* param  *buff - Pointer to the data to be written
* param  sector - Sector address
* param  count - Sector count
* return
*    RES_OK    Read successful
*    STA_NOINIT  Drive not initialized
*    RES_ERROR  Read not successful
******************************************************************************/
DRESULT disk_write(BYTE drv, const BYTE *bufp, DWORD sector, BYTE count) {
  DSTATUS status = RES_OK;
  volatile SD_Params *sdpar = SD_PAR(SD0);

  for (;;) {
    status = RES_NOTRDY;
    if (disk_status(drv) & STA_NOINIT)
      break;

    // Convert LBA to byte address if needed
    if (!(sdpar->HCS))
      sector *= XSDPS_BLK_SIZE_512_MASK;

    status = RES_ERROR;
    if (xsdpsWritePolled(SD0, sector, count, bufp) != RET_OK)
      break;

    status = RES_OK;
    break;  // exit the forever loop
  }
  return status;
}


//-----------------------------------------------------------------------
// Miscellaneous Functions           
//-----------------------------------------------------------------------

DRESULT disk_ioctl(BYTE drv, BYTE ctrl, void *bufp) {
  DRESULT res = 0;
  
  for (;;) {
    res = RES_NOTRDY;  
    if (disk_status(drv) & STA_NOINIT)  // Check if card is in the socket
      break;

    res = RES_ERROR;
    switch (ctrl) {
      case CTRL_SYNC :  // Make sure that no pending write process
        res = RES_OK;
        break;

      case GET_SECTOR_COUNT : // Get number of sectors on the disk (DWORD)
        break;

      case GET_BLOCK_SIZE :  // Get erase block size in unit of sector (DWORD)
        *(DWORD*)bufp = 128;
        res = RES_OK;
        break;

      default:
        res = RES_PARERR;
        break;
    }
    break;  // exit the forever loop
  }
  return res;
}

/*****************************************************************************
* User Provided Timer Function for FatFs module
* param  None
* return  DWORD
****************************************************************************/
DWORD get_fattime(void) {
  return  ((DWORD)(2010 - 1980) << 25)  // Fixed to Jan. 1, 2010
    | ((DWORD)1 << 21)
    | ((DWORD)1 << 16)
    | ((DWORD)0 << 11)
    | ((DWORD)0 << 5)
    | ((DWORD)0 >> 1);
}
