/****************************************************************************
 * sd1player - Audio playback
 *
 * File              : wavefile.cpp
 * This copy created : Apr 20, 2015
 * Version           : 1.13
 * Description       : .wav file format access class (source)
 ***************************************************************************/
#include <string.h>
#include "fatfs.h"
#include "defs.h"
#include "wavefile.h"

typedef struct WaveHdr {
  char id[4];
  unsigned int size;
  char format[4];
  char fmtID[4];
  unsigned int fmtSize;
  unsigned short audioFormat;
  unsigned short numOfChan;
  unsigned int samplesPerSec;
  unsigned int bytesPerSec;
  unsigned short blockAlign;
  unsigned short bitsPerSample;
  char dataID[4];
  unsigned int dataSize;
} WaveHdr;

static short *tempshortBuf = (short *)BUF_CONV;
static void short_to_float(short *sourceBuffer, float *targetBuffer, unsigned int numSamples);

static WaveHdr waveHdr = {
  { 'R','I','F','F' },
  0x24,  // size = total - 8
  { 'W','A','V','E' },
  { 'f','m','t',' ' },
  16,
  1,  // pcm = 1
  1,  // mono
  44100,  // sample rate
  44100*1*(16/8),  // samplerate * numchannels * (bitspersample/8)
  1*(16/8),  // numchannels * (bitspersample/8)
  16,  // bitspersample
  { 'd','a','t','a' },
  0
};

//----------------------------------------------------------------------
// Routine Name : Wavefile
// Input        : none
// Return       : none
// Description  : initialization defaults (constructor)
//----------------------------------------------------------------------
Wavefile::Wavefile() {
  Type(WAVEFILE);
  format = PCM;
}

//----------------------------------------------------------------------
// Routine Name : ~Wavefile
// Input        : none
// Return       : none
// Description  : destructor
//----------------------------------------------------------------------
Wavefile::~Wavefile() {
  f_close(&file);
}

//----------------------------------------------------------------------
// Routine Name : open
// Input        : filename = .wav file to open
//                mode = read or write access
// Return       : error condition
// Description  : virtual open function (source class)
//----------------------------------------------------------------------
int Wavefile::open(char *filename, FileMode mode) {
  unsigned int br;
  WaveHdr waveheader;
  int err;

  err = RET_ERR;
  switch (mode) {
  case FILE_READ:
    if (f_open(&file, filename, FA_READ) != FR_OK)
      break;;

    if (f_read(&file, (unsigned char *)&waveheader, sizeof(WaveHdr), &br) != FR_OK)  // read wave header
      break;;

    track = waveheader.numOfChan == 1 ? MONO : STEREO_BOTH;
    format = (WaveFormat)(waveheader.audioFormat);
    if (waveheader.samplesPerSec != SamplingRate())
      break;
    TotalSamples(waveheader.dataSize / (format == PCM ? sizeof(short) : sizeof(float)));
    Pos(sizeof(WaveHdr));
    err = RET_OK;
    break;
/*
  case FILE_WRITE:
    if (f_open(&file, filename, FA_CREATE_ALWAYS | FA_WRITE) != FR_OK)
      break;
    err = RET_OK;
    break;
*/
  default:
    break;
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : read
// Input        : bufp = destination pointer
//                nBytes = number of bytes to read
// Return       : number of byte read
// Description  : reads a number of raw bytes data
//----------------------------------------------------------------------
int Wavefile::read(unsigned char *bufp, unsigned int nbytes) {
  unsigned int br;
  if (bufp == NULL)
    bufp = (unsigned char *)tempshortBuf;
  if (f_read(&file, (unsigned char *)bufp, nbytes, &br) != FR_OK)
    br = 0;
  return(br);
}

//----------------------------------------------------------------------
// Routine Name : read
// Input        : bufp = destination pointer
//                nSamples = number of samples to read
// Return       : number of byte read
// Description  : reads a number of samples and convert to float
//----------------------------------------------------------------------
int Wavefile::read(float *bufp, unsigned int nsamples) {
  unsigned int br;
  switch (format) {
  case PCM:
    if (f_read(&file, (unsigned char *)tempshortBuf, nsamples*sizeof(short), &br) != FR_OK)
      br = 0;
    else
      short_to_float(tempshortBuf, bufp, (unsigned int)(br/sizeof(short)));
    br /= sizeof(short);
    break;

  case FLOAT:
    if (f_read(&file, (unsigned char *)bufp, nsamples*sizeof(float), &br) != FR_OK)
      br = 0;
    br /= sizeof(float);
    break;
  }
  return(br);
}

//----------------------------------------------------------------------
// Routine Name : write
// Input        : bufp = source pointer
//                nBytes = number of bytes to write
// Return       : error condition
// Description  : writes a number of raw bytes data including the wave file header
//----------------------------------------------------------------------
int Wavefile::write(short *bufp, unsigned int nbytes) {
  UINT totalBw, bw;
  int err;

  for (;;) {
    err = RET_ERR;

    // write the wave header
    if ((err = f_lseek(&file, 0)) != FR_OK)
      break;

    totalBw = nbytes + sizeof(WaveHdr);
    waveHdr.size = totalBw - 8;
    waveHdr.dataSize = totalBw - sizeof(WaveHdr);
    if ((err = f_write(&file, &waveHdr, sizeof(WaveHdr), &bw)) != FR_OK)
      break;
    if ((err = f_write(&file, bufp, totalBw-sizeof(WaveHdr), &bw)) != FR_OK)
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : write
// Input        : bufp = source pointer
//                nSamples = number of samples to write
// Return       : number of samples written or error condition
// Description  : writes a number of samples
//----------------------------------------------------------------------
int Wavefile::write(float *bufp, unsigned int nsamples) {
  UINT bw;
  if (f_write(&file, bufp, nsamples, &bw) != FR_OK)
    return RET_ERR;

  return bw;
}

//----------------------------------------------------------------------
// Routine Name : Pos
// Input        : pos = position to set (in bytes)
// Return       : none
// Description  : file position setter
//----------------------------------------------------------------------
void Wavefile::Pos(unsigned long pos) {
  Source::Pos(pos);
}

//----------------------------------------------------------------------
// Routine Name : seek
// Input        : none
// Return       : current position or error condition
// Description  : actual set position in the file
//----------------------------------------------------------------------
unsigned long Wavefile::seek() {
  unsigned long newPos = Source::Pos();
  unsigned long pos = newPos;
  if (track != MONO)
    pos *= 2;  // stereo
  switch (format) {
  case PCM: pos *= sizeof(short); break;
  case FLOAT: pos *= sizeof(float); break;
  }
  pos +=  sizeof(WaveHdr);
  if (f_lseek(&file, pos) != FR_OK)
    newPos = -1;
  return(newPos);
}

//----------------------------------------------------------------------
// Routine Name : short_to_float
// Input        : src = source pointer 
//                dst = destination pointer
//                nSamples = number of samples to convert
// Return       : none
// Description  : convert a block of audio data from 16bit short to normalized float 
//----------------------------------------------------------------------
static void short_to_float(short *src, float *dst, unsigned int nSamples) {
  for (int i = 0; i < nSamples; i++) {
    float samp = *src * (1.0f / 32768.0f);
    *dst = samp;
    src++;
    dst++;
  }
}
