/*****************************************************
 * mkfilter -- given n, compute recurrence relation
 *             to implement Butterworth of order n
 *             A.J. Fisher, University of York
 *             <fisher@minster.york.ac.uk>
 *             September 1992
 ******************************************************/

#include <stdio.h> 
#include <math.h>
#include "complex.h"
#include "xover.h"
#include "defs.h"

#undef  PI
#define PI 3.14159265358979323846
#define TWOPI (2.0 * PI)
//#define EPS 1e-10  // problem on Cortex-A9 and GCC double
#define EPS 1e-8
#define MAXPZ 2*MAXORDER

struct pzrep {
  complex poles[MAXPZ], zeros[MAXPZ];
  int numpoles, numzeros;
};

static pzrep splane, zplane;
static double raw_alpha1, raw_alpha2;
static double warped_alpha1, warped_alpha2;
static double xcoeffs[MAXPZ+1][MAXTYPES], ycoeffs[MAXPZ+1][MAXTYPES];
static double gain[MAXTYPES];
static int nzeros[MAXTYPES], npoles[MAXTYPES];
static double xv[MAXORDER+1][MAXTYPES*2], yv[MAXORDER+1][MAXTYPES*2];

static int init(int, int);
static float filter(int, int, float);
static void compute_s(int);
static void choosepole(complex);
static void prewarp();
static void normalize(int);
static void compute_z_blt();
static complex blt(complex);
static int expandpoly(int);
static int expand(complex[], int, complex[]);
static void multin(complex, int, complex coeffs[]);

// calculate coefficients for lowpass/highpass/bandpass
int xoverInit(int samplingRate, int cutoff1, int cutoff2, int filterOrder1, int filterOrder2) {
  int err;

  for (;;) {
    raw_alpha1 = (float)(cutoff1)/(float)(samplingRate);
    raw_alpha2 = raw_alpha1;
    if ((err = init(LP, filterOrder1)) != RET_OK)
      break;
  
    if (cutoff2 == 0) {
      if ((err = init(HP, filterOrder1)) != RET_OK)
        break;
    }
    else {
      if ((err = init(BPHP, filterOrder1)) != RET_OK)
        break;

      raw_alpha1 = (float)(cutoff2)/(float)(samplingRate);
      raw_alpha2 = raw_alpha1;

      if ((err = init(BPLP, filterOrder2)) != RET_OK)
        break;
      if ((err = init(HP, filterOrder2)) != RET_OK)
        break;
    }
    break;  // exit the forever loop
  }
  return(err);
}

void xoverClear() {
  for (int i = 0; i < MAXORDER+1; i++) {
    for (int j = 0; j < MAXTYPES*2; j++) {
      xv[i][j] = 0;
      yv[i][j] = 0;
    }
  }
}

float lowpass(int channel, float sample) {
  return(filter(LP, channel, sample));
}

float highpass(int channel, float sample) {
  return(filter(HP, channel, sample));
}

float bandpass(int channel, float sample) {
  return(filter(BPLP, channel, filter(BPHP, channel, sample)));
}

static float filter(int filterType, int channel, float sample) {
  int i, j, numzeros, numpoles;
  double y;

  j = filterType + (channel * MAXTYPES);
  numzeros = nzeros[filterType];
  numpoles = npoles[filterType];

  for (i = 0; i < numzeros; i++)
    xv[i][j] = xv[i+1][j];
  xv[numzeros][j] = sample / gain[filterType];

  for (i = 0; i < numpoles; i++)
    yv[i][j] = yv[i+1][j];

  y = 0.0;
  for (i = 0; i < numzeros+1; i++)
    y += xcoeffs[i][filterType] * xv[i][j];
  
  for (i = 0; i < numpoles; i++)
    y += ycoeffs[i][filterType] * yv[i][j];

  yv[numpoles][j] = y;

  return((float)y);
}

static int init(int filterType, int filterOrder) {
  int err = 0;

  compute_s(filterOrder);
  prewarp();
  normalize(filterType);
  compute_z_blt();
  err = expandpoly(filterType);
  if (err != RET_OK) return(err);
  nzeros[filterType] = zplane.numzeros;
  npoles[filterType] = zplane.numpoles;
  return(RET_OK);
}

// compute S-plane poles for prototype Butterworth filter
static void compute_s(int filterOrder) {
  int i;

  splane.numpoles = 0;
  for (i = 0; i < 2*filterOrder; i++) {
    double theta = (filterOrder & 1) ? (i*PI) / filterOrder : ((i+0.5)*PI) / filterOrder;
    choosepole(expj(theta));
  }
}

static void choosepole(complex z) {
  if (z.re < 0.0)
    splane.poles[splane.numpoles++] = z;
}

// for bilinear transform, perform pre-warp on alpha values
static void prewarp() { 
  warped_alpha1 = tan(PI * raw_alpha1) / PI;
  warped_alpha2 = tan(PI * raw_alpha2) / PI;
}

// called for trad, not for -Re or -Pi
static void normalize(int filterType) {
  int i;
  double w1 = TWOPI * warped_alpha1;
  double w2 = TWOPI * warped_alpha2;

  // transform prototype into appropriate filter type (lp/hp/bp/bs)
  switch (filterType) {
  case LP:
    for (i = 0; i < splane.numpoles; i++) 
      splane.poles[i] = splane.poles[i] * w1;
    splane.numzeros = 0;
    break;

  case HP:
    for (i = 0; i < splane.numpoles; i++) 
      splane.poles[i] = w1 / splane.poles[i];
    for (i = 0; i < splane.numpoles; i++) 
      splane.zeros[i] = 0.0;   // also N zeros at (0,0)
    splane.numzeros = splane.numpoles;
    break;
  
  case BPLP:
    for (i = 0; i < splane.numpoles; i++) 
      splane.poles[i] = splane.poles[i] * w1;
    splane.numzeros = 0;
    break;

  case BPHP:
    for (i = 0; i < splane.numpoles; i++) 
      splane.poles[i] = w1 / splane.poles[i];
    for (i = 0; i < splane.numpoles; i++) 
      splane.zeros[i] = 0.0;   // also N zeros at (0,0)
    splane.numzeros = splane.numpoles;
    break;
  }
}

// given S-plane poles & zeros, compute Z-plane
// poles & zeros by bilinear transform
static void compute_z_blt() {
  int i;

  zplane.numpoles = splane.numpoles;
  zplane.numzeros = splane.numzeros;

  for (i = 0; i < zplane.numpoles; i++)
    zplane.poles[i] = blt(splane.poles[i]);

  for (i = 0; i < zplane.numzeros; i++) 
    zplane.zeros[i] = blt(splane.zeros[i]);

  while (zplane.numzeros < zplane.numpoles) 
    zplane.zeros[zplane.numzeros++] = -1.0;
}

static complex blt(complex pz) {
  return (2.0 + pz) / (2.0 - pz);
}

// given Z-plane poles & zeros, compute top & bot
// polynomials in Z, and then recurrence relation
static int expandpoly(int filterType) {
  int i, err;
  complex fgain;
  complex topcoeffs[MAXPZ+1], botcoeffs[MAXPZ+1];

  err = expand(zplane.zeros, zplane.numzeros, topcoeffs);
  if (err != RET_OK) return(err);
  err = expand(zplane.poles, zplane.numpoles, botcoeffs);
  if (err != RET_OK) return(err);

  switch (filterType) {
  case LP:
    fgain = evaluate(topcoeffs, zplane.numzeros, botcoeffs, zplane.numpoles, 1.0);
    break;
  case HP:
    fgain = evaluate(topcoeffs, zplane.numzeros, botcoeffs, zplane.numpoles, -1.0);
    break;
  case BPLP:
    fgain = evaluate(topcoeffs, zplane.numzeros, botcoeffs, zplane.numpoles, 1.0);
    break;
  case BPHP:
    fgain = evaluate(topcoeffs, zplane.numzeros, botcoeffs, zplane.numpoles, -1.0);
    break;
  }
  gain[filterType] = hypot(fgain);
  
  for (i = 0; i <= zplane.numzeros; i++) 
    xcoeffs[i][filterType] = +(topcoeffs[i].re / botcoeffs[zplane.numpoles].re);
  
  for (i = 0; i <= zplane.numpoles; i++) 
    ycoeffs[i][filterType] = -(botcoeffs[i].re / botcoeffs[zplane.numpoles].re);

  return(RET_OK);
}

// compute product of poles or zeros as a polynomial of z
static int expand(complex pz[], int npz, complex coeffs[]) { 
  int i;
  coeffs[0] = 1.0;

  for (i = 0; i < npz; i++) 
    coeffs[i+1] = 0.0;

  for (i = 0; i < npz; i++) 
    multin(pz[i], npz, coeffs);

  // check computed coeffs of z^k are all real
  for (i = 0; i < npz+1; i++) {
    if (fabs(coeffs[i].im) > EPS)
      return(RET_ERR);
      //fprintf(stderr, "mkfilter: coeff of z^%d is not real; poles/zeros are not complex conjugates\n", i);
  }
  return(RET_OK);
}

// multiply factor (z-w) into coeffs
static void multin(complex w, int npz, complex coeffs[]) { 
  int i;
  complex nw = -w;

  for (i = npz; i >= 1; i--) 
    coeffs[i] = (nw * coeffs[i]) + coeffs[i-1];
  coeffs[0] = nw * coeffs[0];
}
