#include "defs.h"
#include "xparameters.h"
#include "xsdps.h"
#include "cache.h"

#define XSDPS_CMD8_VOL_PATTERN	0x1AA
#define XSDPS_RESPOCR_READY	0x80000000
#define XSDPS_ACMD41_HCS	0x40000000
#define XSDPS_ACMD41_3V3	0x00300000
#define XSDPS_CMD1_HIGH_VOL	0x00FF8000

#define XSDPS_SCR_BLKCNT	        1
#define XSDPS_SCR_BLKSIZE	        8
#define XSDPS_4_BIT_WIDTH	        0x2
#define XSDPS_SWITCH_CMD_BLKCNT		1
#define XSDPS_SWITCH_CMD_BLKSIZE	64
#define XSDPS_SWITCH_CMD_HS_GET	        0x00FFFFF0
#define XSDPS_SWITCH_CMD_HS_SET	        0x80FFFFF1
#define XSDPS_EXT_CSD_CMD_BLKCNT	1
#define XSDPS_EXT_CSD_CMD_BLKSIZE	512
#define XSDPS_INIT_DELAY	        2

static int cmdTransfer(int sdNum, UINT32 cmd, UINT32 arg, UINT32 bCount);
static UINT32 frameCmd(UINT32 Cmd);
static void setupADMA2DescTbl(int sdNum, UINT32 bCount, const UINT8 *bufp);

static void SD_IRQHandler(int sdNum);
static void SD0_IRQHandler(void* param) { SD_IRQHandler(SD0); }

static void SD0_IRQHandler(void* param);
static volatile SD_Params sd0Par;
  
SD_PORT_T SD_Port[TOTAL_SD_PORT] = {
#ifdef MICROZED
  { (volatile SD_Type *)XPAR_PS7_SD_0_BASEADDR, XPAR_XSDIOPS_0_INTR, SD0_IRQHandler, (SD_Params *)&sd0Par }  // SD0
#else
  { (volatile SD_Type *)XPAR_PS7_SD_1_BASEADDR, XPAR_XSDIOPS_1_INTR, SD0_IRQHandler, (SD_Params *)&sd0Par }  // SD1
#endif
};


// ---------------------------------------------------------------------------
UINT32 SR;
static void SD_IRQHandler(int sdNum) {
  volatile SD_Type *sd = SD_REG(sdNum);
  SR = sd->NORM_INTR_STS;
  sd->NORM_INTR_STS = SR;
}

// not accurate, just some delay
void delayUs(UINT32 us) {
  for (volatile int i = 0; i < us; i++)
    for (volatile int j = 0; j < 78; j++)
      ;
}

int xsdpsIsInitialized(int sdNum) {
  volatile SD_Type *sd = SD_REG(sdNum);
  return ((sd->CLK_CTRL & XSDPS_CC_SD_CLK_EN_MASK) == 0 ? FALSE : TRUE);
}

//----------------------------------------------------------------------
// Routine Name : xsdpsInitialize
// Input        : sdNum = SD id
// Return       : error status
// Description  : initializes the host controller
//                Initial clock of 400KHz is set
//                Voltage of 3.3V is selected as that is supported by host
//                Interrupts status is enabled and signal disabled by default
//                Default data direction is card to host and
//                32 bit ADMA2 is selected. Default Block size is 512 bytes
//----------------------------------------------------------------------
int xsdpsInitialize(int sdNum) {
  volatile SD_Type *sd = SD_REG(sdNum);

  sd->SW_RST = XSDPS_SWRST_ALL_MASK;  // "Software reset for all" is initiated
  while (sd->SW_RST & XSDPS_SWRST_ALL_MASK) // wait for reset complete
    ;

  // SD clock frequency divider 128. Enable the internal clock
  sd->CLK_CTRL = XSDPS_CC_SDCLK_FREQ_D128_MASK | XSDPS_CC_INT_CLK_EN_MASK;
  // Wait for internal clock to stabilize
  while ((sd->CLK_CTRL & XSDPS_CC_INT_CLK_STABLE_MASK) == 0)
    ;
  sd->CLK_CTRL |= XSDPS_CC_SD_CLK_EN_MASK;  // enable SD clock

  // select voltage and enable bus power
  sd->POWER_CTRL = XSDPS_PC_BUS_VSEL_3V3_MASK | XSDPS_PC_BUS_PWR_MASK;
  sd->HOST_CTRL1 = XSDPS_HC_DMA_ADMA2_32_MASK;

  // enable all interrupt status except card interrupt initially
  sd->NORM_INTR_STS_EN = XSDPS_NORM_INTR_ALL_MASK & (~XSDPS_INTR_CARD_MASK);
  sd->ERR_INTR_STS_EN = XSDPS_ERROR_INTR_ALL_MASK;

  // disable all interrupt signals by default
  sd->NORM_INTR_SIG_EN = 0x0; //XSDPS_INTR_TC_MASK; // except Transfer Complete
  sd->ERR_INTR_SIG_EN = 0x0;

  // transfer mode register - default value
  // DMA enabled, block count enabled, data direction card to host(read)
  sd->XFER_MODE = XSDPS_TM_DMA_EN_MASK | XSDPS_TM_BLK_CNT_EN_MASK | XSDPS_TM_DAT_DIR_SEL_MASK;

  sd->BLK_SIZE = XSDPS_BLK_SIZE_512_MASK;  // set block size to 512 by default

  return RET_OK;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsSdCardInitialize
// Input        : sdNum = SD id
// Return       : error status
// Description  : initializes the SD card by following its initialization
//                identification state diagram
//                CMD0 is sent to reset card
//                CMD8 and ACDM41 are sent to identify voltage and high capacity support
//                CMD2 and CMD3 are sent to obtain Card ID and Relative card address respectively
//                CMD9 is sent to read the card specific data
//----------------------------------------------------------------------
int xsdpsSdCardInitialize(int sdNum) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);
  volatile SD_Params *sdpar = SD_PAR(sdNum);

  err = RET_ERR;
  for (;;) {
    // check the present state register to make sure card is inserted and detected by host controller
    //if ((sd->PRES_STATE & XSDPS_PSR_CARD_INSRT_MASK) == 0)
    //  break;
    // 74 CLK delay after card is powered up, before the first command
    delayUs(XSDPS_INIT_DELAY);

    if (cmdTransfer(sdNum, CMD0, 0, 0) != RET_OK) // CMD0 no response expected
      break;

    // CMD8; response expected
    // 0x1AA - Supply Voltage 2.7 - 3.6V and AA is pattern
    if (cmdTransfer(sdNum, CMD8, XSDPS_CMD8_VOL_PATTERN, 0) != RET_OK)
      break;
    if (sd->RESP0 != XSDPS_CMD8_VOL_PATTERN)
      break;

    // send ACMD41 while card is still busy with power up
    UINT32 respOCR;
    status = RET_OK;
    do {
      if ((status = cmdTransfer(sdNum, CMD55, 0, 0)) != RET_OK)
        break;

      // 0x40300000 - Host High Capacity support & 3.3V window
      if ((status = cmdTransfer(sdNum, ACMD41, (XSDPS_ACMD41_HCS | XSDPS_ACMD41_3V3), 0)) != RET_OK)
        break;
      respOCR = sd->RESP0;
    } while ((respOCR & XSDPS_RESPOCR_READY) == 0); // response with card capacity
    if (status != RET_OK)
      break;

    // update HCS support flag based on card capacity response
    sdpar->HCS = (respOCR & XSDPS_ACMD41_HCS) ? 1 : 0;

    if (cmdTransfer(sdNum, CMD2, 0, 0) != RET_OK)  // CMD2 for Card ID
      break;

    do {
      if ((status = cmdTransfer(sdNum, CMD3, 0, 0)) != RET_OK)
	break;
      // relative card address is stored as the upper 16 bits
      // this is to avoid shifting when sending commands
      sdpar->RelCardAddr = sd->RESP0 & 0xFFFF0000;
    } while (sdpar->RelCardAddr == 0);
    if (status != RET_OK)
      break;

    if (cmdTransfer(sdNum, CMD9, (sdpar->RelCardAddr), 0) != RET_OK)
      break;

    // card specific data is read (currently not used for any operation)
    UINT32 dummyRead;
    dummyRead = sd->RESP0; dummyRead = sd->RESP1; dummyRead = sd->RESP2; dummyRead = sd->RESP3;
  
    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsReadPolled
// Input        : sdNum = SD id
//                arg = address passed by the user that is to be sent as
//                argument along with the command
//                bCount = Block count passed by the user
//                bufp = Pointer to the data buffer for a DMA transfer
// Return       : error status
// Description  : performs SD read in polled mode
//----------------------------------------------------------------------
int xsdpsReadPolled(int sdNum, UINT32 arg, UINT32 bCount, UINT8 *bufp) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);

  err = RET_ERR;
  for (;;) {
    // Check status to ensure card is initialized
    //if ((sd->PRES_STATE & XSDPS_PSR_CARD_INSRT_MASK) == 0x0)
    //  break;

    // Set block size to 512 if not already set
    if (sd->BLK_SIZE != XSDPS_BLK_SIZE_512_MASK) {
      if (xsdpsSetBlockSize(sdNum, XSDPS_BLK_SIZE_512_MASK) != RET_OK)
	break;
    }
    setupADMA2DescTbl(sdNum, bCount, bufp);
    sd->XFER_MODE = XSDPS_TM_AUTO_CMD12_EN_MASK |
                    XSDPS_TM_BLK_CNT_EN_MASK |
                    XSDPS_TM_DAT_DIR_SEL_MASK |
                    XSDPS_TM_DMA_EN_MASK |
                    XSDPS_TM_MUL_SIN_BLK_SEL_MASK;
    Xil_DCacheInvalidateRange((unsigned int)bufp, bCount * XSDPS_BLK_SIZE_512_MASK);

    if (cmdTransfer(sdNum, CMD18, arg, bCount) != RET_OK)  // Send block read command
      break;
    
    // Check for transfer complete
    UINT32 statusReg;
    status = RET_OK;
    do {
      if ((statusReg = sd->NORM_INTR_STS) & XSDPS_INTR_ERR_MASK) {
        status = sd->ERR_INTR_STS;
        sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK; // Write to clear error bits
	break;
      }
    } while ((statusReg & XSDPS_INTR_TC_MASK) == 0);
    if (status != RET_OK)
      break;

    sd->NORM_INTR_STS = XSDPS_INTR_TC_MASK;  // Write to clear bit
    status = sd->RESP0;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsWritePolled
// Input        : sdNum = SD id
//                arg = address passed by the user that is to be sent as
//                argument along with the command
//                bCount = Block count passed by the user
//                bufp = Pointer to the data buffer for a DMA transfer
// Return       : error status
// Description  : performs SD write in polled mode
//----------------------------------------------------------------------
int xsdpsWritePolled(int sdNum, UINT32 arg, UINT32 bCount, const UINT8 *bufp) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);

  err = RET_ERR;
  for (;;) {
    // Check status to ensure card is initialized
    //if ((sd->PRES_STATE & XSDPS_PSR_CARD_INSRT_MASK) == 0x0)
    //  break;

    // Set block size to 512 if not already set
    if (sd->BLK_SIZE != XSDPS_BLK_SIZE_512_MASK ) {
      if (xsdpsSetBlockSize(sdNum, XSDPS_BLK_SIZE_512_MASK) != RET_OK)
	break;
    }
    setupADMA2DescTbl(sdNum, bCount, bufp);
    Xil_DCacheFlushRange((unsigned int)bufp, bCount * XSDPS_BLK_SIZE_512_MASK);
    sd->XFER_MODE = XSDPS_TM_AUTO_CMD12_EN_MASK |
                    XSDPS_TM_BLK_CNT_EN_MASK |
                    XSDPS_TM_MUL_SIN_BLK_SEL_MASK |
                    XSDPS_TM_DMA_EN_MASK;

    if (cmdTransfer(sdNum, CMD25, arg, bCount) != RET_OK)  // Send block write command
      break;

    // Check for transfer complete
    UINT32 statusReg;
    status = RET_OK;
    do {
      if ((statusReg = sd->NORM_INTR_STS) & XSDPS_INTR_ERR_MASK) {
        sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK;  // Write to clear error bits
        status = RET_ERR;
        break;
      }
    } while ((statusReg & XSDPS_INTR_TC_MASK) == 0);
    if (status != RET_OK)
      break;

    sd->NORM_INTR_STS = XSDPS_INTR_TC_MASK;  // Write to clear bit

    err = RET_OK;
    break;
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsSelectCard
// Input        : sdNum = SD id
// Return       : error status
// Description  : selects card and sets default block size
//----------------------------------------------------------------------
int xsdpsSelectCard (int sdNum) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);
  volatile SD_Params *sdpar = SD_PAR(sdNum);


  err = RET_ERR;
  for (;;) {
    // Send CMD7 - Select card
    if (cmdTransfer(sdNum, CMD7, sdpar->RelCardAddr, 0) != RET_OK)
      break;
    status = sd->RESP0;

    if (xsdpsSetBlockSize(sdNum, XSDPS_BLK_SIZE_512_MASK) != RET_OK) // Set default block size
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : cmdTransfer
// Input        : sdNum = SD id
//                arg = argument to be sent along with the command
//                bCount = Block count passed by the user
// Return       : error status
// Description  : SD command generation
//----------------------------------------------------------------------
static int cmdTransfer(int sdNum, UINT32 cmd, UINT32 arg, UINT32 bCount) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);

  err = RET_ERR;
  for (;;) {
    // Check the command inhibit to make sure no other command transfer is in progress
    if (sd->PRES_STATE & XSDPS_PSR_INHIBIT_CMD_MASK)
      break;
   
    sd->BLK_CNT = bCount;  // Write block count register
    sd->TIMEOUT_CTRL = 0xE;
    
    sd->ARGMT = arg;  // Write argument register
    sd->NORM_INTR_STS = XSDPS_NORM_INTR_ALL_MASK;
    sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK;

    // Command register is set to trigger transfer of command
    // Mask to avoid writing to reserved bits 31-30
    // This is necessary because 0x80000000 is used  by this software to
    // distinguish between ACMD and CMD of same number
    UINT32 cmdReg = frameCmd(cmd) & 0x3FFF;

    // Check for data inhibit in case of command using DAT lines
    if ((sd->PRES_STATE & XSDPS_PSR_INHIBIT_CMD_MASK) && (cmdReg & XSDPS_DAT_PRESENT_SEL_MASK))
      break;
    sd->CMD = cmdReg;

    // Polling for response
    UINT32 statusReg;
    status = RET_OK;
    do {
      if ((statusReg = sd->NORM_INTR_STS) & XSDPS_INTR_ERR_MASK) {
        status = sd->ERR_INTR_STS;
        sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK;  // Write to clear error bits
	break;
      }
    } while((statusReg & XSDPS_INTR_CC_MASK) == 0);
    if (status != RET_OK)
      break;
    sd->NORM_INTR_STS = XSDPS_INTR_CC_MASK;    // Write to clear bit

    err = RET_OK;
    break; // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : frameCmd
// Input        : Cmd = Command to be sent
// Return       : Command register value complete with response type and
//                data, CRC and index related flags
// Description  : frames the Command register for a particular command
//                generates only the command register value i.e. the upper 16 bits of the
//                transfer mode and command register. This value is already shifted to be
//                upper 16 bits and can be directly OR'ed with transfer mode register value
//----------------------------------------------------------------------
static UINT32 frameCmd(UINT32 Cmd) {
    UINT32 RetVal;

    RetVal = Cmd;
    switch(Cmd) {
    case CMD0: RetVal |= RESP_NONE; break;
    case CMD1: RetVal |= RESP_R3; break;
    case CMD2: RetVal |= RESP_R2; break;
    case CMD3: RetVal |= RESP_R6; break;
    case CMD4: RetVal |= RESP_NONE; break;
    case CMD5: RetVal |= RESP_R1B;  break;
    case CMD6: RetVal |= RESP_R1 | XSDPS_DAT_PRESENT_SEL_MASK; break;
    case ACMD6: RetVal |= RESP_R1; break;
    case CMD7: RetVal |= RESP_R1; break;
    case CMD8: RetVal |= RESP_R1; break;
    case CMD9: RetVal |= RESP_R2; break;
    case CMD10:
    case CMD12:
    case ACMD13:
    case CMD16: RetVal |= RESP_R1; break;
    case CMD17:
    case CMD18: RetVal |= RESP_R1 | XSDPS_DAT_PRESENT_SEL_MASK; break;
    case CMD23:
    case ACMD23:
    case CMD24:
    case CMD25: RetVal |= RESP_R1 | XSDPS_DAT_PRESENT_SEL_MASK;
    case ACMD41: RetVal |= RESP_R3; break;
    case ACMD42: RetVal |= RESP_R1; break;
    case ACMD51: RetVal |= RESP_R1 | XSDPS_DAT_PRESENT_SEL_MASK; break;
    case CMD52:
    case CMD55: RetVal |= RESP_R1; break;
    case CMD58:
    break;
    }
    return RetVal;
}

//----------------------------------------------------------------------
// Routine Name : setupADMA2DescTbl
// Input        : sdNum = SD id
//                bCount = block count
//                bufp = pointer to data buffer
// Return       : none
// Description  : API to setup ADMA2 descriptor table
//----------------------------------------------------------------------
static void setupADMA2DescTbl(int sdNum, UINT32 bCount, const UINT8 *bufp) {
  UINT32 totalDescLines = 0;
  UINT32 descNum = 0;
  UINT32 bSize = 0;
  volatile SD_Type *sd = SD_REG(sdNum);
  volatile SD_Params *sdpar = SD_PAR(sdNum);

  // Setup ADMA2 - Write descriptor table and point ADMA SAR to it
  bSize = sd->BLK_SIZE & XSDPS_BLK_SIZE_MASK;
  if ((bCount * bSize) < XSDPS_DESC_MAX_LENGTH) {
    totalDescLines = 1;
  }
  else {
    totalDescLines = ((bCount * bSize) / XSDPS_DESC_MAX_LENGTH);
    if ((bCount * bSize) % XSDPS_DESC_MAX_LENGTH)
      totalDescLines += 1;
  }
  for (descNum = 0; descNum < (totalDescLines-1); descNum++) {
    sdpar->Adma2_DescrTbl[descNum].Address = (UINT32)(bufp + (descNum * XSDPS_DESC_MAX_LENGTH));
    sdpar->Adma2_DescrTbl[descNum].Attribute = XSDPS_DESC_TRAN | XSDPS_DESC_VALID;
    // write '0' to length field which indicates 65536
    sdpar->Adma2_DescrTbl[descNum].Length = (UINT16)XSDPS_DESC_MAX_LENGTH;
  }
  sdpar->Adma2_DescrTbl[totalDescLines-1].Address = (UINT32)(bufp + (descNum * XSDPS_DESC_MAX_LENGTH));
  sdpar->Adma2_DescrTbl[totalDescLines-1].Attribute = XSDPS_DESC_TRAN | XSDPS_DESC_END | XSDPS_DESC_VALID;
  sdpar->Adma2_DescrTbl[totalDescLines-1].Length = (bCount * bSize) - (descNum * XSDPS_DESC_MAX_LENGTH);
  sd->ADMA_SAR = (UINT32)&(sdpar->Adma2_DescrTbl[0]);
  Xil_DCacheFlushRange((unsigned int)(&(sdpar->Adma2_DescrTbl[0])), sizeof(Adma2Descr) * 32);
}

//----------------------------------------------------------------------
// Routine Name : xsdpsSetBlockSize(int sdNum, UINT16 bSize) {
// Input        : sdNum = SD id
//                bSize = Block size passed by the user
// Return       : none
// Description  : Update Block size for read/write operations
//----------------------------------------------------------------------
int xsdpsSetBlockSize(int sdNum, UINT16 bSize) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);

  err = RET_ERR;
  for (;;) {
    if (sd->PRES_STATE & (XSDPS_PSR_INHIBIT_CMD_MASK | XSDPS_PSR_INHIBIT_DAT_MASK |
                          XSDPS_PSR_WR_ACTIVE_MASK | XSDPS_PSR_RD_ACTIVE_MASK))
      break;

    if (cmdTransfer(sdNum, CMD16, bSize, 0) != RET_OK)  // Send block write command
      break;
    status = sd->RESP0;
    sd->BLK_SIZE = bSize & XSDPS_BLK_SIZE_MASK; // Set block size to the value passed

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsGetBusWidth
// Input        : sdNum = SD id
//                SCR = buffer to store SCR register returned by card
// Return       : error status
// Description  : API to get bus width support by card
//----------------------------------------------------------------------
int xsdpsGetBusWidth(int sdNum, UINT8 *SCR) {
  UINT32 err, status;
  UINT16 bCount;
  volatile SD_Type *sd = SD_REG(sdNum);
  volatile SD_Params *sdpar = SD_PAR(sdNum);

  err = RET_ERR;
  for (;;) {
    for (int i = 0; i < 8; i++) {
      SCR[i] = 0;
    }
    // Send block write command
    if (cmdTransfer(sdNum, CMD55, sdpar->RelCardAddr, 0) != RET_OK)
      break;

    sd->BLK_SIZE = XSDPS_SCR_BLKSIZE & XSDPS_BLK_SIZE_MASK;
    bCount = XSDPS_SCR_BLKCNT;
    setupADMA2DescTbl(sdNum, bCount, SCR);
    sd->XFER_MODE = XSDPS_TM_DAT_DIR_SEL_MASK | XSDPS_TM_DMA_EN_MASK;
    Xil_DCacheInvalidateRange((unsigned int)SCR, 8);
    if (cmdTransfer(sdNum, ACMD51, 0, bCount) != RET_OK)
      break;

    // Check for transfer complete (Polling for response for now)
    UINT32 statusReg;
    status = RET_OK;
    do {
      if ((statusReg = sd->NORM_INTR_STS) & XSDPS_INTR_ERR_MASK) {        
        sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK; // Write to clear error bits
        status = RET_ERR;
        break;
      }
    } while ((statusReg & XSDPS_INTR_TC_MASK) == 0);
    if (status != RET_OK)
      break;
 
    sd->NORM_INTR_STS = XSDPS_INTR_TC_MASK; // Write to clear bit
    status = sd->RESP0;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsChangeBusWidth
// Input        : sdNum = SD id
// Return       : error status
// Description  : API to set bus width to 4-bit in card and host
//----------------------------------------------------------------------
int xsdpsChangeBusWidth(int sdNum) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);
  volatile SD_Params *sdpar = SD_PAR(sdNum);

  err = RET_ERR;
  for (;;) {
    if (cmdTransfer(sdNum, CMD55, sdpar->RelCardAddr, 0) != RET_OK)
      break;

    if (cmdTransfer(sdNum, ACMD6, XSDPS_4_BIT_WIDTH, 0) != RET_OK)
      break;

    sd->HOST_CTRL1 |= XSDPS_HC_WIDTH_MASK;
    status = sd->RESP0;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsGetBusSpeed
// Input        : sdNum = SD id
//                bufp = buffer to store function group support data returned by card
// Return       : error status
// Description  : API to get bus speed supported by card
//----------------------------------------------------------------------
int xsdpsGetBusSpeed(int sdNum, UINT8 *bufp) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);

  err = RET_ERR;
  for (;;) {
    for (int i = 0; i < 64; i++) {
      bufp[i] = 0;
    }
    sd->BLK_SIZE = XSDPS_SWITCH_CMD_BLKSIZE & XSDPS_BLK_SIZE_MASK;
    setupADMA2DescTbl(sdNum, XSDPS_SWITCH_CMD_BLKCNT, bufp);
    sd->XFER_MODE = XSDPS_TM_DAT_DIR_SEL_MASK | XSDPS_TM_DMA_EN_MASK;
    Xil_DCacheInvalidateRange((unsigned int)bufp, 64);
    if (cmdTransfer(sdNum, CMD6, XSDPS_SWITCH_CMD_HS_GET, 1) != RET_OK)
      break;

    // Check for transfer complete (Polling for response for now)
    UINT32 statusReg;
    status = RET_OK;
    do {
      if ((statusReg = sd->NORM_INTR_STS) & XSDPS_INTR_ERR_MASK) {
        sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK; // Write to clear error bits
        status = RET_ERR;
        break;
      }
    } while ((statusReg & XSDPS_INTR_TC_MASK) == 0);
    if (status != RET_OK)
      break;

    sd->NORM_INTR_STS = XSDPS_INTR_TC_MASK;  // Write to clear bit
    status = sd->RESP0;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsChangeBusSpeed
// Input        : sdNum = SD id
// Return       : error status
// Description  : API to set high speed in card and host. Changes clock in host accordingly
//----------------------------------------------------------------------
int xsdpsChangeBusSpeed(int sdNum) {
  UINT32 err, status;
  volatile SD_Type *sd = SD_REG(sdNum);

  err = RET_ERR;
  for (;;) {
    UINT8 rbuf[64] __attribute__ ((aligned(32)));
    sd->BLK_SIZE = XSDPS_SWITCH_CMD_BLKSIZE & XSDPS_BLK_SIZE_MASK;
    setupADMA2DescTbl(sdNum, XSDPS_SWITCH_CMD_BLKCNT, rbuf);
    Xil_DCacheFlushRange((unsigned int)rbuf, 64);
    sd->XFER_MODE = XSDPS_TM_DAT_DIR_SEL_MASK | XSDPS_TM_DMA_EN_MASK;
    if (cmdTransfer(sdNum, CMD6, XSDPS_SWITCH_CMD_HS_SET, 1) != RET_OK)
      break;
    // Check for transfer complete (Polling for response for now)
    UINT32 statusReg;
    status = RET_OK;
    do {
      if ((statusReg = sd->NORM_INTR_STS) & XSDPS_INTR_ERR_MASK) {
        sd->ERR_INTR_STS = XSDPS_ERROR_INTR_ALL_MASK;  // Write to clear error bits
        status = RET_ERR;
        break;
      }
    } while ((statusReg & XSDPS_INTR_TC_MASK) == 0);
    if (status != RET_OK)
      break;
  
    sd->NORM_INTR_STS = XSDPS_INTR_TC_MASK;  // Write to clear bit
    sd->CLK_CTRL &= ~(XSDPS_CC_INT_CLK_EN_MASK | XSDPS_CC_SD_CLK_EN_MASK);

    UINT32 clockReg;
    clockReg = sd->CLK_CTRL & (~XSDPS_CC_SDCLK_FREQ_SEL_MASK);
    clockReg |= XSDPS_CC_SDCLK_FREQ_BASE_MASK | XSDPS_CC_INT_CLK_EN_MASK;
    sd->CLK_CTRL = clockReg;
    // Wait for internal clock to stabilize
    while ((sd->CLK_CTRL & XSDPS_CC_INT_CLK_STABLE_MASK) == 0)
      ;
    // Enable SD clock   
    sd->CLK_CTRL |= XSDPS_CC_SD_CLK_EN_MASK;
    sd->HOST_CTRL1 |= XSDPS_HC_SPEED_MASK;
    status = sd->RESP0;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsChangeClockFreq
// Input        : sdNum = SD id
//                selFreq = Clock frequency in Hz
// Return       : none
// Description  : change clock frequency to the value less than or equal
//                to the given value using the permissible dividors
//----------------------------------------------------------------------
int xsdpsChangeClockFreq(int sdNum, UINT32 selFreq) {
  UINT16 divisor;
  int status, err;
  volatile SD_Type *sd = SD_REG(sdNum);
  volatile SD_Params *sdpar = SD_PAR(sdNum);

  err = RET_ERR;
  for (;;) {
    // Disable clock
    sd->CLK_CTRL &= ~(XSDPS_CC_INT_CLK_EN_MASK | XSDPS_CC_SD_CLK_EN_MASK);

    // Calculate divisor
    int divCount = 0x1;
    int i;
    for (i = 0; i < XSDPS_CC_MAX_NUM_OF_DIV; i++) {
#ifdef MICROZED
      if (((XPAR_PS7_SD_0_SDIO_CLK_FREQ_HZ) / divCount) <= selFreq) {
#else
      if (((XPAR_PS7_SD_1_SDIO_CLK_FREQ_HZ) / divCount) <= selFreq) {
#endif
        divisor = (divCount / 2) << XSDPS_CC_DIV_SHIFT;
        break;
      }
      divCount = divCount << 1;
    }
    if (i == 9)
      break; // No valid divisor found for given frequency
 
    // Set clock divisor
    UINT16 clockReg = sd->CLK_CTRL & (~XSDPS_CC_SDCLK_FREQ_SEL_MASK);
    clockReg |= divisor | XSDPS_CC_INT_CLK_EN_MASK;
    sd->CLK_CTRL = clockReg;
  
    // Wait for internal clock to stabilize
    while ((sd->CLK_CTRL & XSDPS_CC_INT_CLK_STABLE_MASK) == 0)
      ;
    sd->CLK_CTRL |= XSDPS_CC_SD_CLK_EN_MASK;  // Enable SD clock

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}

//----------------------------------------------------------------------
// Routine Name : xsdpsPullup
// Input        : sdNum = SD id
// Return       : error status
// Description  : API to send pullup command to card before using DAT line 3(using 4-bit bus)
//----------------------------------------------------------------------
int xsdpsPullup(int sdNum) {
  UINT32 err;
  volatile SD_Params *sdpar = SD_PAR(sdNum);

  err = RET_ERR;
  for (;;) {
    if (cmdTransfer(sdNum, CMD55, sdpar->RelCardAddr, 0) != RET_OK)
      break;

    if (cmdTransfer(sdNum, ACMD42, 0, 0) != RET_OK)
      break;

    err = RET_OK;
    break;  // exit the forever loop
  }
  return err;
}
