----------------------------------------------------------------------------------
-- Mike Field (hamster@snap.net.nz)
-- Create Date:    20:04:52 09/06/2011 
-- Module Name:    clock_recovery - Behavioral 
----------------------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity clock_recovery is
  Port(
    clk : in  STD_LOGIC;
    transition : in  STD_LOGIC;
    sampleNow : out STD_LOGIC
  );
end clock_recovery;

architecture Behavioral of clock_recovery is
  type reg is record
    pulseLen : std_logic_vector(6 downto 0);
    minPulse : std_logic_vector(6 downto 0);
    validCount : std_logic_vector(6 downto 0);
    pulse1pt5 : std_logic_vector(7 downto 0);
    pulse2pt5 : std_logic_vector(8 downto 0);
    valid : std_logic;
    sampleNow : std_logic;
  end record;
	
  signal r : reg := ((others => '0'),(others => '0'),(others => '0'),(others => '0'),(others => '0'),'0','0');
  signal n : reg;

begin
  sampleNow <= r.sampleNow;
  process(r, transition)
  begin
    n <= r;

    n.pulse1pt5 <= ("00" & r.minPulse(6 downto 1)) + r.minPulse;
    n.pulse2pt5 <= ("00" & r.minPulse(6 downto 1)) + ("0"&r.minPulse&"0");
    
    -- output the recovered sample clock
    n.sampleNow <= '0';
    -- is it time to sample (0.5 pulse)?
    if r.pulseLen = "0" & r.minPulse(6 downto 1) then
       n.sampleNow <= r.valid;
    end if;

    -- is it time to sample (1.5 pulse)?
    if ("0" & r.pulseLen) =  r.pulse1pt5 then
       n.sampleNow <= r.valid;
    end if;

    -- is it time to sample (2.5 pulse)?
    if ("0" & r.pulseLen) = r.pulse2pt5 then
       n.sampleNow <= r.valid;
    end if;
    
    if transition = '1' then
      -- start timing the pulse and check if we have a new minimum
      n.pulseLen <= (others => '0');
      if r.pulseLen < r.minPulse then
        -- if this is the shortest pulse so far then we need to remove the valid bit
        n.minPulse <= r.pulseLen;
        n.validCount <= (others => '0');
        n.valid <= '0';
      else
        -- we need 255 valid pulses before we declare that the recovered clock is valid
        if r.validCount = "1111111" then
          n.valid <= '1';
        else
          n.validCount <= r.validCount + 1;
          n.valid <= '0';
        end if;
      end if;
    else
      -- if the pulse is 2.55us long it's invalid
      if r.pulseLen = "1111111" then
        n.pulseLen <= r.pulseLen;        
        n.validCount <= (others => '0');
        n.valid <= '0';
      else
        n.pulseLen <= r.pulseLen + 1;
      end if;
      
      -- if the pulse is longer than 4x the minimum pulse length it's invalid
      if r.pulseLen > (r.minPulse & "00") then
        n.minPulse <= r.pulseLen;
        n.validCount <= (others => '0');
        n.valid <= '0';
      end if;
    end if;
  end process;
  
  process(r, clk)
  begin
    if rising_edge(clk) then
      r <= n;
    end if;
  end process;

end Behavioral;
