----------------------------------------------------------------------
-- Engineer: Mike Field (hamster@snap.net.nz) 
-- 
-- Create Date:    16:23:13 09/04/2011 
-- Module Name:    spdif_in - Behavioral 
--
-- Description: Reclocks and then samples the incoming S/PDIF signal.
----------------------------------------------------------------------
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity spdif_in is
  Port(
    clk : in STD_LOGIC;
    spdif_in : in STD_LOGIC;
    sampleNow : in std_logic;
    transition : out std_logic;
    subframe : out STD_LOGIC_VECTOR (27 downto 0);
    channelA : out STD_LOGIC;
    startOfFrame : out STD_LOGIC;
    newSubFrame : out STD_LOGIC
  );
end spdif_in;

architecture Behavioral of spdif_in is
  type reg is record
    reclock : std_logic_vector(2 downto 0);
    bits : std_logic_vector(127 downto 0);
    newSubFrame : std_logic;
    lastNewSubFrame: std_logic;
    startOfFrame : std_logic;
    channelA : std_logic;
  end record;
   
  signal r : reg := ((others => '0'),(others => '0'), '0', '0', '0', '0');
  signal n : reg;
   
begin
  -- Set output signals
  startOfFrame <= r.startOFFrame;
  channelA <= r.channelA;
  newSubFrame <= r.newSubFrame AND NOT r.lastNewSubFrame;
  transition <= r.reclock(1) XOR r.reclock(0);
   
  -- map the bits from the shift register into the sample outputs
  g: for i in 0 to 27 generate
  begin
    subFrame( i) <= r.bits(8+i*2) xor r.bits(9+i*2);
  end generate;
   
  process(r, spdif_in, sampleNow)
  begin
    n <= r;
    n.lastNewSubframe <= r.newSubFrame;
      
    -- synchronize the input signal, and also allow an easy way to look for transitions
    n.reclock <= spdif_in & r.reclock(2 downto 1);
      
    -- have we just got a valid frame?
      
    -- do we need to take a sample of the input signal
    if sampleNow = '1' then
      n.bits <= r.reclock(0) & r.bits(127 downto 1);
    end if;

    n.newSubFrame <= '0';
    n.startOfFrame <= '0';
    n.channelA <= '0';
    -- is the first subframe's preamble either of the "channel a" patterns?
    if r.bits(8 downto 0) = "100010111" or r.bits(8 downto 0) = "011101000" then
      n.startOfFrame <= '1';
      n.channelA <= '1';
      n.newSubFrame <= '1';
    elsif r.bits(8 downto 0) = "101000111" or r.bits(8 downto 0) = "010111000" then
      n.channelA <= '1';
      n.newSubFrame <= '1';
    elsif r.bits(8 downto 0) = "100100111" or r.bits(8 downto 0) = "011011000" then
      n.newSubFrame <= '1';
    end if;
  end process;
   
  process(clk)
  begin
    if rising_edge(clk) then
      r <= n;
    end if;
  end process;
end Behavioral;
