library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

entity spdif_data is port(
  spdif_clk : IN std_logic; -- 128x Fsample
  reset : IN std_logic;
  trigger_arm : IN std_logic;
  trigger_pol : IN std_logic;
  trigger_chan : IN std_logic;
  trigger : OUT std_logic;
  pause : IN std_logic;
  mute : IN std_logic;
  BRAM0_addr : OUT std_logic_vector(31 downto 0);
  BRAM0_data : IN std_logic_vector(31 downto 0);
  BRAM1_addr : OUT std_logic_vector(31 downto 0);
  BRAM1_data : IN std_logic_vector(31 downto 0);
  stereo_sample : OUT std_logic_vector(31 downto 0);
  Flip : OUT std_logic_vector(7 downto 0)
);
end entity spdif_data;

architecture behavioral of spdif_data is
  signal addr : std_logic_vector(17 downto 0) := (others => '0');  -- BRAM address
  signal bit_counter : std_logic_vector(6 downto 0) := (others => '0');  -- spdif bit counter (stereo)
  signal stereo_sample_s : std_logic_vector(31 downto 0) := (others => '0');  -- left & right signed 16-bit sample
 
  -- mute and fading
  signal muteCount : integer := 0;
  signal lastMute : std_logic;
  signal lastSignLeft : std_logic;
  signal lastCountLeft : integer := 0;
  signal lastSignRight : std_logic;
  signal lastCountRight : integer := 0;
   
  -- trigger
  signal trigger_s : std_logic := '0';  -- latch trigger state
  signal trigger_flipped : std_logic := '0';  -- trigger flip has occurred
  signal trigger_count : std_logic_vector(1 downto 0) := (others => '0');  -- number of nulls to count on trigger zero
    
begin
  BRAM0_addr <= x"0000" & '0' & addr(14 downto 0);
  BRAM1_addr <= x"0000" & '0' & addr(14 downto 0); 
  Flip <= "00001" & addr(17 downto 15);  -- indicates BRAM0/BRAM1 flip to PS
  stereo_sample <= stereo_sample_s;
  trigger <= trigger_s;

  -- 64 counts for left sample (channelA) and 64 counts for right sample
  spdif_clk_counter : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then 
        bit_counter <= (others => '0');
      else
        bit_counter <= bit_counter + 1;
      end if;
    end if;
  end process spdif_clk_counter;

  fading_ramp : process (spdif_clk)
  variable data : std_logic_vector(15 downto 0);
  variable data_fading : std_logic_vector(15 downto 0);
  variable data_step : std_logic_vector(15 downto 0);
  variable count : integer;
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if lastMute /= mute then  -- apply fading ramp when mute change
        muteCount <= 0;
        lastMute <= mute;
        lastCountLeft <= 0;
        lastCountRight <= 0;
      end if;
      
      if bit_counter = "0000011" then
        if addr(15) = '0' then
          stereo_sample_s <= BRAM0_data;
        else
          stereo_sample_s <= BRAM1_data;
        end if;
      end if;
                
      if bit_counter(6 downto 1) = "000010" then  -- at ticks 4 (left sample) & 5 (right sample)
        if bit_counter(0) = '0' then
          data := stereo_sample_s(15 downto 0);  -- left sample
          if mute = '1' then
            stereo_sample_s(15 downto 0) <= x"0000";
          end if;
        else
          data := stereo_sample_s(31 downto 16);  -- right sample
          if mute = '1' then
            stereo_sample_s(31 downto 16) <= x"0000";
          end if;
        end if;
            
        data_step := x"0000";
        -- 1/8 of sample
        if data(15) = '0' then
          data_fading := "000" & data(15 downto 3);
        else
          data_fading := x"ffff" - data;
          data_fading := "000" & data_fading(15 downto 3);
        end if;
 
        if muteCount < 40000 then
          -- decreasing/increasing in 8 steps of 5000 samples each
          muteCount <= muteCount + 1;    
          count := 1;
          for i in 0 to 7 loop
            if muteCount > count then
              if bit_counter(0) = '0' then
                -- left sample
                if count <= lastCountLeft then
                  data_step := data_step + data_fading;
                else
                  -- wait to cross zero before moving to next step
                  if lastSignLeft /= data(15) then
                    lastCountLeft <= count;
                  end if;
                end if;
              else
                -- right sample
                if count <= lastCountRight then
                  data_step := data_step + data_fading;
                else
                  -- wait to cross zero before moving to next step
                  if lastSignRight /= data(15) then
                    lastCountRight <= count;
                  end if;
                end if;
              end if;
            end if;
            count := count + 5000; -- step length
          end loop;
          -- apply the ramp fading value to the sample
          if mute = '1' then
            if data(15) = '0' then
              data := data - data_step;
            else
              data := data + data_step;
            end if;
          else
            if data(15) = '0' then
              data := data_step;
            else
              data := x"ffff" - data_step;
            end if;      
          end if;
          if bit_counter(0) = '0' then
            stereo_sample_s(15 downto 0) <= data;  -- left sample
          else
            stereo_sample_s(31 downto 16) <= data;  -- right sample
          end if;
        end if;
        if bit_counter(0) = '0' then
          lastSignLeft <= data(15);
        else
          lastSignRight <= data(15);
        end if;
      end if;
    end if;
  end process fading_ramp;
   
  inc_addr : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then 
        addr <= (others => '0');
      else
        if bit_counter = "0000111" then
          if pause = '0' then
            addr <= addr+4;  -- next stereo sample
          end if;
        end if;
      end if;
    end if;
  end process inc_addr;
  
  -- when trigger is armed from PS, set trigger for the first non-zero sample
  -- trigger has to be desarmed from PS before enabling the cycle again
  set_trigger : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then 
        trigger_flipped <= '0';
        trigger_s <= '0';
        trigger_count <= "00";
      else
        trigger_s <= '0';
        if trigger_arm = '1' then
          if trigger_pol = '1' then
            -- non-zero sample
            if stereo_sample_s /= x"00000000" then
              if trigger_flipped = '0' then
                trigger_flipped <= '1';
                trigger_s <= '1';
              end if;
            end if;
          else
            -- 3 consecutiove zero samples
            if  bit_counter(6) /= trigger_chan and bit_counter(5 downto 0) = "000100" then
              if stereo_sample_s = x"00000000" then
                if trigger_flipped = '0' then
                  if trigger_count = "10" then
                    trigger_flipped <= '1';
                    trigger_s <= '1';
                  else
                    trigger_count <= trigger_count + 1;
                  end if;
                end if;
              else
                trigger_count <= "00";
              end if;
            end if;
          end if;    
        else
          trigger_flipped <= '0';  -- trigger disarmed from PS
        end if;
      end if;
    end if;
  end process set_trigger;
              
end behavioral;
