-- https://ackspace.nl/wiki/SP/DIF_transmitter_project
-- From Hackerspace ACKspace
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

entity spdif_out is 
generic (
  C_DELAY_LEFT : std_logic_vector(9 downto 0);
  C_DELAY_RIGHT : std_logic_vector(9 downto 0)
);
Port (
  sys_clk : IN  STD_LOGIC;
  spdif_clk : IN std_logic;
  reset : IN std_logic;
  to_fir_valid : OUT std_logic;
  from_fir_valid : IN std_logic;
  from_fir_left : IN std_logic_vector(15 downto 0);
  from_fir_right : IN std_logic_vector(15 downto 0);
  addra : out STD_LOGIC_VECTOR ( 10 downto 0 );
  dina : out STD_LOGIC_VECTOR ( 15 downto 0 );
  douta : in STD_LOGIC_VECTOR ( 15 downto 0 );
  wea : out STD_LOGIC_VECTOR ( 0 to 0 );
  spdif : out  STD_LOGIC
);
end spdif_out;

architecture Behavioral of spdif_out is
  signal bit_counter : std_logic_vector(6 downto 0) := (others => '0');  -- bit6 marks channelA (when =0)
  signal frame_counter : std_logic_vector(8 downto 0) := (others => '0');
  signal channel_status_shift : std_logic_vector(23 downto 0);
  signal channel_status : std_logic_vector(23 downto 0) := "001000000000000001000000";
  
  signal data_in_buffer : std_logic_vector(23 downto 0);
  signal data_out_buffer : std_logic_vector(7 downto 0):= (others => '0');
  signal data_biphase : std_logic := '0';
  signal parity : std_logic;
  
  signal dataL_from_fir : std_logic_vector(15 downto 0);
  signal dataR_from_fir : std_logic_vector(15 downto 0);
     
  signal adrL_wr : std_logic_vector(9 downto 0) := (others => '0');
  signal adrR_wr : std_logic_vector(9 downto 0) := (others => '0');
  signal adr_rd : std_logic_vector(9 downto 0) := (others => '0');
  signal inc_adr : std_logic_vector(2 downto 0) := (others => '0');
  signal dataL_bram : std_logic_vector(15 downto 0);
  signal dataR_bram : std_logic_vector(15 downto 0);
  
begin

  write_fir : process (sys_clk)
  begin
    if sys_clk'event and sys_clk = '0' then   
      to_fir_valid <= '0';
      if bit_counter = "0001001" then
        to_fir_valid <= '1';
      end if;    
    end if;
  end process write_fir;    

  read_fir : process (sys_clk)
  begin
    if sys_clk'event and sys_clk = '0' then
      if reset = '1' then
        adrL_wr <= C_DELAY_LEFT;
        adrR_wr <= C_DELAY_RIGHT;
        adr_rd <= "0000000000";
        inc_adr <= (others => '0');
      else
        wea(0) <= '0';
        if from_fir_valid = '1' then
          dataL_bram <= from_fir_left;  -- save fir output
          dataR_bram <= from_fir_right;
                
          addra <= '0' & adrL_wr;  -- left samples on bram lower 512 addresses
          dina <= dataL_bram;
          wea(0) <= '1';  -- write left sample into bram
          inc_adr <= "001";  -- state machine start
        else
          case inc_adr is
            when "001" =>
              addra <= '1' & adrR_wr;  -- right samples on bram higher 512 addresses
              dina <= dataR_bram;
              wea(0) <= '1';  -- write right sample into bram 
            when "010" =>
              addra <= '0' & adr_rd; 
            --when "011" =>
            when "100" =>
              dataL_from_fir <= douta;  -- read left sample from bram
              addra <= '1' & adr_rd;   
            --when "101" =>
            when "110" =>
              dataR_from_fir <= douta;  -- read right sample from bram
            when "111" =>            
              adrL_wr <= adrL_wr + 1;
              adrR_wr <= adrR_wr + 1;
              adr_rd <= adr_rd + 1;
            when others =>
          end case;
          if inc_adr /= "000" then
            inc_adr <= inc_adr + 1;
          end if; 
        end if;
      end if;
    end if;
  end process read_fir;

  spdif_clk_counter : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then 
        bit_counter <= (others => '0');
      else
        bit_counter <= bit_counter + 1;
      end if;
    end if;
  end process spdif_clk_counter;

  data_latch : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then 
        frame_counter <= (others => '0');
      else
        parity <= data_in_buffer(23) xor data_in_buffer(22) xor data_in_buffer(21) xor data_in_buffer(20) xor data_in_buffer(19) xor data_in_buffer(18) xor data_in_buffer(17)  xor data_in_buffer(16) xor data_in_buffer(15) xor data_in_buffer(14) xor data_in_buffer(13) xor data_in_buffer(12) xor data_in_buffer(11) xor data_in_buffer(10) xor data_in_buffer(9) xor data_in_buffer(8) xor data_in_buffer(7) xor data_in_buffer(6) xor data_in_buffer(5) xor data_in_buffer(4) xor data_in_buffer(3) xor data_in_buffer(2) xor data_in_buffer(1) xor data_in_buffer(0) xor channel_status_shift(23);

        if bit_counter = "0000011" then
          data_in_buffer <= dataL_from_fir & x"00";
        end if;
        if bit_counter = "1000011" then
          data_in_buffer <= dataR_from_fir & x"00";
        end if;
        if bit_counter(5 downto 0) = "111111" then
          if frame_counter = "101111111" then
            frame_counter <= (others => '0');
          else
            frame_counter <= frame_counter + 1;
          end if;
        end if;
      end if;
    end if;
  end process data_latch;
 
  data_output : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then 
        channel_status <= "001000000000000001000000";
        data_out_buffer <= (others => '0');
      else
        if bit_counter(5 downto 0) = "111111" then
          if frame_counter = "101111111" then -- next frame is 0, load preamble Z (B)
            channel_status_shift <= channel_status;
            data_out_buffer <= "10011100";
          else
            if frame_counter(0) = '1' then -- next frame is even, load preamble X (M)
              channel_status_shift <= channel_status_shift(22 downto 0) & '0';
              data_out_buffer <= "10010011";
            else -- next frame is odd, load preable Y (W)
              data_out_buffer <= "10010110";
            end if;
          end if;
        else
          if bit_counter(2 downto 0) = "111" then -- load new part of data into buffer
            case bit_counter(5 downto 3) is
            when "000" =>
              data_out_buffer <= '1' & data_in_buffer(0) & '1' & data_in_buffer(1) & '1' & data_in_buffer(2) & '1' & data_in_buffer(3);
            when "001" =>
              data_out_buffer <= '1' & data_in_buffer(4) & '1' & data_in_buffer(5) & '1' & data_in_buffer(6) & '1' & data_in_buffer(7);
            when "010" =>
              data_out_buffer <= '1' & data_in_buffer(8) & '1' & data_in_buffer(9) & '1' & data_in_buffer(10) & '1' & data_in_buffer(11);
            when "011" =>
              data_out_buffer <= '1' & data_in_buffer(12) & '1' & data_in_buffer(13) & '1' & data_in_buffer(14) & '1' & data_in_buffer(15);
            when "100" =>
              data_out_buffer <= '1' & data_in_buffer(16) & '1' & data_in_buffer(17) & '1' & data_in_buffer(18) & '1' & data_in_buffer(19);
            when "101" =>
              data_out_buffer <= '1' & data_in_buffer(20) & '1' & data_in_buffer(21) & '1' & data_in_buffer(22) & '1' & data_in_buffer(23);
            when "110" =>
              data_out_buffer <= "10101" & channel_status_shift(23) & "1" & parity;
            when others =>
            end case;
          else
            data_out_buffer <= data_out_buffer(6 downto 0) & '0';
          end if;
        end if;
      end if;
    end if;
  end process data_output;
  
  biphaser : process (spdif_clk)
  begin
    if spdif_clk'event and spdif_clk = '1' then
      if reset = '1' then
        data_biphase <= '0';
      else
        if data_out_buffer(data_out_buffer'left) = '1' then
          data_biphase <= not data_biphase;
        end if;
      end if;
    end if;
  end process biphaser;
 
  spdif <= data_biphase;

end Behavioral;
