/* Epiphany-specific functions */
#include <string.h>
#include <ncurses.h>

#include <sys/time.h>
#include <time.h>

#include <e-hal.h>

#include "epiphany.h"
#include "main.h"

#define MODES   16
#define FUNCBUF 8192

/* epiphany device */
static e_epiphany_t *dev;

/* function name fetching */
static char funcbuf[FUNCBUF];
static char *unknown = "-";
static const char *tool = "/opt/adapteva/esdk/tools/e-gnu/bin/e-addr2line";

/* timer control */
static struct ep_core cores[CORES];
static struct ep_tmode_text modes[MODES] = {
	{ 'o', "timer Off"                 },
	{ 'c', "Clock cycles"              },
	{ 'i', "Idle cycles"               },
	{ '3', "Reserved (3)"              },
	{ 'a', "iAlu instructions"         },
	{ 'f', "Fpu instructions"          },
	{ 'd', "Dual issue cycles"         },
	{ 'e', "stall: E1 (load)"          },
	{ 'r', "stall: RA (register dep.)" },
	{ '9', "Reserved (9)"              },
	{ 'f', "stall: Fetch (local)"      },
	{ 'l', "stall: Load (local)"       },
	{ 'F', "stall: Fetch (external)"   },
	{ 'L', "stall: Load  (external)"   },
	{ 'm', "Mesh traffic monitor 0"    },
	{ 'M', "Mesh traffic monitor 1"    },
};

/* select epiphany device */
void select_epiphany(e_epiphany_t *d)
{
	dev = d;
}

/* print header */
void print_header(void)
{
	struct timeval tv;
	gettimeofday(&tv, NULL);
	struct tm *time = localtime(&tv.tv_sec);

	/* move cursor to top left */
	xmove(0, 0);

	/* print current time */
	xclrtoeol();
	xprintf("e-top v%i - %02d:%02d:%02d\n",
		VERSION,
		time->tm_hour, time->tm_min, time->tm_sec);

	if(!opt_batch) {
		xprintf("Press 'h' or '?' for help.\n");
	}

	/* print header bar */
	xclrtoeol();
	xattron(A_BOLD | A_REVERSE);
	xprintf(" %2s %7s %4s %3s %1s %2s %13s %13s %9s %s",
		"ID", "CORE", " ST ", "FPU", "I", "EX",
		"TIMER 0", "TIMER 1", "PC", "FUNC");

	if(!opt_batch) {	/* fill the line */
		int x, y, mx, my;
		getyx(stdscr, y, x);
		getmaxyx(stdscr, my, mx);
		(void)y; (void)my;
		for(int i = 0; i < mx-x-1; i++)
			xprintf(" ");
	}
	xprintf("\n");
	xattroff(A_BOLD | A_REVERSE);
}

/* acquire data from epiphany */
void update_states()
{
	/* state variables */
	uint32_t coreid[CORES];
	uint32_t config[CORES];
	uint32_t status[CORES];
	uint32_t dstatus[CORES];
	uint32_t ctimer0[CORES];
	uint32_t ctimer1[CORES];
	uint32_t pc[CORES];
	uint32_t ctimer_max = 0xFFFFFFFF;
	uint32_t cmd;

	/* read registers, update if enabled */
	for(int i = 0; i < CORES_X; i++) {
		for(int j = 0; j < CORES_Y; j++) {
			int c = i * CORES_Y + j;

			/* read epiphany registers */
			e_read(dev, i, j, E_REG_COREID,
				&coreid[c], sizeof(uint32_t));
			e_read(dev, i, j, E_REG_CONFIG,
				&config[c], sizeof(uint32_t));
			e_read(dev, i, j, E_REG_STATUS,
				&status[c], sizeof(uint32_t));
			e_read(dev, i, j, E_REG_DEBUGSTATUS,
				&dstatus[c], sizeof(uint32_t));
			e_read(dev, i, j, E_REG_CTIMER0,
				&ctimer0[c], sizeof(uint32_t));
			e_read(dev, i, j, E_REG_CTIMER1,
				&ctimer1[c], sizeof(uint32_t));
			e_read(dev, i, j, E_REG_PC,
				&pc[c], sizeof(uint32_t));

			/* reset timer values if enabled */
			if(cores[c].timers[0]) {
				e_write(dev, i, j, E_REG_CTIMER0,
					&ctimer_max, sizeof(uint32_t));
			}

			if(cores[c].timers[1]) {
				e_write(dev, i, j, E_REG_CTIMER1,
					&ctimer_max, sizeof(uint32_t));
			}
		}
	}

	/* parse core states */
	for(int c = 0; c < CORES; c++) {
		/* get coordinates */
		int i = c / CORES_Y;
		int j = c % CORES_Y;

		/* row and col */
		cores[c].row = (coreid[c] & 0xFC0) >> 6;
		cores[c].col = (coreid[c] & 0x03F);
//		cores[c].row = i;
//		cores[c].col = j;

		/* state */
		if(dstatus[c] & 0x1) {
			/* halted: check for TRAP */
			uint16_t instr;
			e_read(dev, i, j, (off_t)(pc[c]-2),
				&instr, sizeof(instr));
			if((instr & 0x3FF) == 0x3E2) {	/* TRAP */
				cores[c].state = EP_TRAP;
				cores[c].trap  = (instr&0xFC00) >> 10;
			} else {
				cores[c].state = EP_HALT;
			}
		} else {
			/* not halted */
			if(status[c] & 0x1)
				cores[c].state = EP_RUNNING;
			else
				cores[c].state = EP_IDLE;
		}

		/* fpu mode */
		if(config[c] & 0x80000)
			cores[c].fpu = EP_INTEGER;
		else
			cores[c].fpu = EP_FLOAT;

		/* interrupt flag */
		if(status[c] & 0x2)
			cores[c].interrupts = 0;
		else
			cores[c].interrupts = 1;

		/* exception cause */
		cores[c].excause = (status[c] & 0x000F0000) >> 16;

		/* timer modes / values */
		cores[c].tmodes[0]  = (config[c] & 0x0F0) >> 4;
		cores[c].tvalues[0] = ctimer0[c];
		cores[c].tmodes[1]  = (config[c] & 0xF00) >> 8;
		cores[c].tvalues[1] = ctimer1[c];

		/* program counter and function name */
		cores[c].pc       = pc[c];
		cores[c].funcname = unknown;
		cores[c].srcline  = unknown;
	}

	/* fetch function names */
	if(opt_numelfs == 1) {			/* single ELF file */
		/* build command line with all addresses */
		char cmd[512];
		int len = 0;
		len += snprintf(cmd, 512, "%s -e %s -C -s -f",
			tool, opt_elfs[0]);
		for(int c = 0; c < CORES; c++)
			len += snprintf(cmd+len, 512-len,
				" 0x%x", cores[c].pc);

		/* run command */
		FILE* pipe = popen(cmd, "r");
		if(!pipe) {	/* error: ignore all */
			strncpy(funcbuf, "(error)", FUNCBUF);
			for(int c = 0; c < CORES; c++)
				cores[c].funcname = funcbuf;
			return;
		}

		/* parse output */
		len = 0;
		for(int c = 0; c < CORES; c++) {
			/* function name */
			if(fgets(funcbuf+len, FUNCBUF-len, pipe)) {
				char *tmp = strstr(funcbuf+len, "\n");
				if(tmp)
					*tmp = '\0';
				cores[c].funcname = funcbuf+len;
				len += strlen(funcbuf+len) + 1;
			}

			/* source line */
			if(fgets(funcbuf+len, FUNCBUF-len, pipe)) {
				char *tmp = strstr(funcbuf+len, "\n");
				if(tmp)
					*tmp = '\0';
				cores[c].srcline = funcbuf+len;
				len += strlen(funcbuf+len) + 1;
			}
		}
		pclose(pipe);
	} else if(opt_numelfs == CORES) {	/* per-core ELF files */
		int len = 0;
		for(int c = 0; c < CORES; c++) {
			/* build command line */
			char cmd[512];
			snprintf(cmd, 512, "%s -e %s -C -s -f 0x%x",
				tool, opt_elfs[c], cores[c].pc);

			/* run command */
			FILE* pipe = popen(cmd, "r");
			if(!pipe)	/* error: skip core */
				continue;

			/* parse output */
			if(fgets(funcbuf+len, FUNCBUF-len, pipe)) {
				char *tmp = strstr(funcbuf+len, "\n");
				if(tmp)
					*tmp = '\0';
				cores[c].funcname = funcbuf+len;
				len += strlen(funcbuf+len) + 1;
			}

			if(fgets(funcbuf+len, FUNCBUF-len, pipe)) {
				char *tmp = strstr(funcbuf+len, "\n");
				if(tmp)
					*tmp = '\0';
				cores[c].srcline = funcbuf+len;
				len += strlen(funcbuf+len) + 1;
			}

			pclose(pipe);
		}
	}
}

/* print states */
void print_states(e_epiphany_t *dev)
{
	uint32_t tval;

	for(int c = 0; c < CORES; c++) {
		/* clear until end of line */
		xclrtoeol();

/* ID */	xprintf(" %2d ", c);
/* CORE */	xprintf("[%2d,%2d] ", cores[c].row, cores[c].col);

/* STATE */	switch(cores[c].state) {
		case EP_RUNNING:
			xattron(COLOR_PAIR(3));
			xprintf("  R  ");
			xattroff(COLOR_PAIR(3));
			break;
		case EP_IDLE:
			xattron(COLOR_PAIR(4));
			xprintf("  I  ");
			xattroff(COLOR_PAIR(4));
			break;
		case EP_HALT:
			xattron(COLOR_PAIR(5));
			xprintf("  H  ");
			xattroff(COLOR_PAIR(5));
			break;
		case EP_TRAP:
			xattron(COLOR_PAIR(5));
			xprintf(" T%02d ", cores[c].trap);
			xattroff(COLOR_PAIR(5));
			break;
		}

/* FPU */	switch(cores[c].fpu) {
		case EP_FLOAT:
			xattron(COLOR_PAIR(4));
			xprintf(" F ");
			xattroff(COLOR_PAIR(4));
			break;
		case EP_INTEGER:
			xattron(COLOR_PAIR(2));
			xprintf(" I ");
			xattroff(COLOR_PAIR(2));
			break;
		}

/* INT */	if(cores[c].interrupts) {
			xattron(COLOR_PAIR(4));
			xprintf(" +");
			xattroff(COLOR_PAIR(4));
		} else {
			xattron(COLOR_PAIR(2));
			xprintf(" -");
			xattroff(COLOR_PAIR(2));
		}

/* EX */	xattron(COLOR_PAIR(cores[c].excause != 0 ? 2 : 4));
		xprintf(" %02d ", cores[c].excause);
		xattroff(COLOR_PAIR(cores[c].excause != 0 ? 2 : 4));

/* TIM0 */	tval = cores[c].tvalues[0];
		xprintf("%c%c:",
			(cores[c].timers[0]) ? '!' : ' ',
			modes[cores[c].tmodes[0]].letter);
		if(tval == 0) {
			xattron(COLOR_PAIR(3));
			xprintf("%10s ", "MAX    ");
			xattroff(COLOR_PAIR(3));
		} else {
			xprintf("%10u ", 0xFFFFFFFF - tval);
		}

/* TIM1 */	tval = cores[c].tvalues[1];
		xprintf("%c%c:",
			(cores[c].timers[1]) ? '!' : ' ',
			modes[cores[c].tmodes[1]].letter);
		if(tval == 0) {
			xattron(COLOR_PAIR(3));
			xprintf("%10s ", "MAX    ");
			xattroff(COLOR_PAIR(3));
		} else {
			xprintf("%10u ", 0xFFFFFFFF - tval);
		}

/* PC */	xattron(COLOR_PAIR(cores[c].pc < 0x10000 ? 4 : 2));
		xprintf(" %8x ", cores[c].pc);
		xattroff(COLOR_PAIR(cores[c].pc < 0x10000 ? 4 : 2));

/* FUNC */	/* non-batch: don't write more than there is space */
		if(opt_batch) {
			xprintf("%s (%s)",
				cores[c].funcname, cores[c].srcline);
		} else {
			xprintf(" ");
			int x, y, mx, my, rem;
			getyx(stdscr, y, x);
			getmaxyx(stdscr, my, mx);
			(void)y; (void)my;

			rem = mx - x - 4;
			char tmp[16];
			if(rem > 1) {
				snprintf(tmp, 16, "%%-.%ds ", rem);
				xprintf(tmp, cores[c].funcname);
			}

			rem -= strlen(cores[c].funcname);
			if(rem > 1) {
				snprintf(tmp, 16, "(%%-.%ds)", rem);
				xprintf(tmp, cores[c].srcline);
			}
		}

		xprintf("\n");
	}

	xprintf("\n");
}

void ep_show_actions(void)
{
	xprintf("Use the following commands to take over the timer:\n");
	xprintf("  X     ENABLE  e-top handling of timer\n");
	xprintf("  x     DISABLE e-top handling of timer\n");
	xprintf("\n");

	xprintf("To select the timer clock source, use the following keys:\n");
	for(size_t i = 0; i < MODES; i++) {
		xprintf("  %c\t%s\n", modes[i].letter, modes[i].string);
	}
}

int ep_action(int core, int timer, int key)
{
	switch(key) {
	case 'X':
		cores[core].timers[timer] = 1;
		return(1);
	case 'x':
		cores[core].timers[timer] = 0;
		return(1);
	}

	for(size_t m = 0; m < MODES; m++) {
		if(key == modes[m].letter) {
			uint32_t cmd;
			uint32_t config;
			int i = core % CORES_X;
			int j = core / CORES_X;

			/* stop the core */
			cmd = 0x1;
			e_write(dev, i, j, E_REG_DEBUGCMD,
				&cmd, sizeof(uint32_t));

			/* read config register */
			e_read(dev, i, j, E_REG_CONFIG,
				&config, sizeof(uint32_t));

			/* change timer mode bits */
			if(timer == 0) {
				config &= 0xFFFFFF0F;
				config |= m << 4;
			} else {
				config &= 0xFFFFF0FF;
				config |= m << 8;
			}

			/* write config register */
			e_write(dev, i, j, E_REG_CONFIG,
				&config, sizeof(uint32_t));

			/* resume core */
			cmd = 0x0;
			e_write(dev, i, j, E_REG_DEBUGCMD,
				&cmd, sizeof(uint32_t));

			xrefresh();
			xdelay(0.3);
			return(1);
		}
	}

	return(0);
}
