/* Handling of user input */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ncurses.h>

#include "main.h"
#include "epiphany.h"

/* NOTE: input is restricted to interactive mode;
         these functions may use terminal functions freely */

/* helper functions
   ======================================================================== */
static float read_float()
{
	char buf[32];
	nodelay(stdscr, FALSE); echo();
	getnstr(buf, 32);
	noecho(); nodelay(stdscr, TRUE);
	return(atof(buf));
}

static int read_int()
{
	char buf[32];
	int  val;
	nodelay(stdscr, FALSE); echo();
	getnstr(buf, 32);
	noecho(); nodelay(stdscr, TRUE);

	val = atoi(buf);
	if(val == 0 && strncmp(buf, "0", 32)) {
		return(-1);
	} else {
		return(val);
	}
}

static int read_key()
{
	nodelay(stdscr, FALSE);
	int key = getch();
	nodelay(stdscr, TRUE);
	return(key);
}

/* menu action functions
   ======================================================================== */
static void set_opt_secs()
{
	float new;

	xattron(A_BOLD);
	xprintf("  Change delay from %.1f to: ", opt_secs);
	xattroff(A_BOLD);
	new = read_float();
	if(new > 0 && new <= 60) {
		opt_secs = new;
	} else {
		xprintf("  Invalid number.\n");
		xrefresh();
		xdelay(1);
		xclear();
	}
}

/* menu screens
   ======================================================================== */
static void menu_timers()
{
	static int selected_timer = 0;
	static int selected_core  = 0;
	int num;

show_menu_timers:
	xclear();
	xattron(A_BOLD);
	xprintf("Epiphany Timer Menu\n\n");
	xattroff(A_BOLD);

	xprintf("Selected core:  ");
	xattron(A_BOLD | A_REVERSE);
	xprintf(" %2i ", selected_core);
	xattroff(A_BOLD | A_REVERSE);
	xprintf(" (press s to select core)\n");

	xprintf("Selected timer: ");
	xattron(A_BOLD | A_REVERSE);
	xprintf(" %2i ", selected_timer);
	xattroff(A_BOLD | A_REVERSE);
	xprintf(" (press 0 or 1 to select timer)\n\n");

	ep_show_actions();

	xprintf("\nPress 'q' to return, or type a command key.\n");
	int key = read_key();
	switch(key) {
	case 'q': return;	/* quit to main */

	case '0':		/* select timer 0 and try again */
		selected_timer = 0;
		goto show_menu_timers;
	case '1':		/* select timer 1 and try again */
		selected_timer = 1;
		goto show_menu_timers;
	case 's':
		xattron(A_BOLD);
		xprintf("Select new core (currently %i): ", selected_core);
		xattroff(A_BOLD);
		num = read_int();
		if(num < 0 || num >= CORES) {
			xprintf("Invalid core.\n");
			xrefresh();
			xdelay(1);
		} else {
			selected_core = num;
		}
		goto show_menu_timers;

	default:
		if(ep_action(selected_core, selected_timer, key)) {
			/* key was handled by epiphany system */
			return;
		} else {
			/* unknown key, try generic handler */
			handle_key(key);
		}
	return;
	}
}

static void menu_help()
{
	xclear();
	xattron(A_BOLD);
	xprintf("Help for Interactive Commands - e-top v%i\n\n", VERSION);
	xattroff(A_BOLD);

	xprintf("Main commands:\n");
	xprintf("  h or ?    Show this help screen\n");
	xprintf("  e         Main screen explanation\n");
	xprintf("  d         Change update delay\n");
	xprintf("  q         Quit program\n");
	xprintf("\n");

	xprintf("Epiphany commands:\n");
	xprintf("  t         Show TIMER menu\n");

	/* now wait for a key and handle it */
	xprintf("\nPress 'q' to return, or type a command key.\n");
	int key = read_key();
	if(key != 'q')
		handle_key(key);
	return;
}

static void menu_explanation()
{
	xclear();
	xattron(A_BOLD);
	xprintf("Main Screen Explanation Page\n");
	xattroff(A_BOLD);
	xprintf("\n");
	xprintf("The columns have the following meaning:\n");
	xprintf("  ID, CORE    core identifier and on-chip coordinates\n");
	xprintf("  STate       [R]unning, [I]dle, [H]alted, Trapped (Txx)\n");
	xprintf("  FPU mode    signed [I]nteger or [F]loat\n");
	xprintf("  Interrupts  on [+] or off [-]\n");
	xprintf("  EXcause     EXCAUSE register value\n");
	xprintf("  TIMER 0/1   CTIMER clock source and value (**)\n");
	xprintf("  PC          program counter value\n");
	xprintf("  FUNC        currently executing function (***)\n");
	xprintf("\n");
	xprintf(" (*)  For more details, type 't' to see the Timer Menu.\n");
	xprintf(" (**) Requires ELF file names on command line and uses\n");
	xprintf("      e-addr2line program. Compile ELF files with debug\n");
	xprintf("      information ('-g' option). Quite slow.\n");
	xprintf("\n");
	xprintf("Some debug output may be shown below the main screen.\n");

	xprintf("\nPress 'q' to return, 't' for timer menu,\n"
		"or any other key to return to the main help.\n");
	switch(read_key()) {
	case 't': menu_timers(); break;
	case 'q': return;
	default:  menu_help(); break;
	}
}

/* main key handler
   ======================================================================== */
int handle_key(int key)
{
	switch(key) {
	/* Main Commands */
	case 'h':
	case '?':
		menu_help();
		break;
	case 'e':
		menu_explanation();
		break;
	case 'd':
		set_opt_secs();
		break;
	case 'q':
		/* handled by main() */
		return(0);

	/* Epiphany Commands */
	case 't':
		menu_timers();
		break;

	default:
		xprintf("  unknown key: %i\n", key);
		xrefresh();
		xdelay(1);
		break;
	}
	xclear();

	return(1);
}
