/* Option parsing functions */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include "main.h"

static void usage(char *prog)
{
	printf("\n");
	printf("Usage:\n");
	printf("  %s [-h] [-b] [-n num] [-d secs] [elfs...]\n", prog);
	printf("Options:\n");
	printf("    -h:      print help and exit\n");
	printf("    -b:      enter batch mode\n");
	printf("    -n num:  print 'num' updates, then exit\n");
	printf("    -d secs: update screen every 'secs' seconds (float)\n");
	printf("\n");
	printf("    elfs:    currently running ELF file(s)\n");
	printf("        ONE filename:  single, shared ELF file\n");
	printf("        %3i filenames: per-core ELF file\n", CORES);
	printf("\n");
}

/* parse command line arguments */
int parse_opts(int argc, char **argv)
{
	int opt_char;

	opterr = 0;
	while((opt_char = getopt(argc, argv, "hbn:d:")) != -1) {
		switch(opt_char) {
		case 'h':
			usage(argv[0]);
			return(1);
		case 'b':
			opt_batch = 1;
			break;
		case 'n':
			opt_num  = atoi(optarg);
			if(opt_num < 0) {
				fprintf(stderr, "Invalid number for -n.\n");
				return(1);
			}
			break;
		case 'd':
			opt_secs = atof(optarg);
			if(opt_secs <= 0) {
				fprintf(stderr, "Invalid number for -d.\n");
				return(1);
			}
			break;
		case '?':
			switch(optopt) {
			case 'n':
			case 'd':
				fprintf(stderr, "Missing option for -%c.\n",
					optopt);
				break;
			default:
				fprintf(stderr, "Invalid option -%c.\n",
					optopt);
				break;
			}
			return(1);
		case ':':
			fprintf(stderr, "Error parsing options.\n");
			return(1);
		}
	}

	/* check number of non-option arguments (ELF files) */
	switch(argc - optind) {
	case 0:		/* no ELF files given */
		opt_numelfs = 0; break;
	case 1:		/* shared ELF file */
		opt_numelfs = 1; break;
	case CORES:	/* per-core ELF files */
		opt_numelfs = CORES; break;
	default:	/* invalid */
		fprintf(stderr,
			"Please specify exactly ONE or %i ELF files!\n",
			CORES);
		return(1);
	}

	/* open each file to check validity */
	for(int idx = 0; idx < (argc - optind); idx++) {
		opt_elfs[idx] = argv[idx + optind];
		FILE *file = fopen(opt_elfs[idx], "r");
		if(!file) {
			fprintf(stderr, "Could not open '%s': %s\n",
				opt_elfs[idx], strerror(errno));
			return(1);
		}
		fclose(file);
	}

	return(0);
}

