/* Epiphany Host Application */

#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include "e-hal.h"
#include "../shared.h"

#define FAIL(...) { fprintf(stderr, __VA_ARGS__); exit(1); }
#define SHM_OFFSET 0x01000000
extern void handle_event(shm_t *);
extern uint32_t vtop(uint32_t vaddr);

/* files to load into Epiphany (top-left to bottom-right) */
char *files[] = {
	"bin/test.srec",
	"", "", "", "", "", "", "", "", "",
};

int main(int argc, char *argv[])
{
	shm_t        shm;
	e_epiphany_t dev;
	e_mem_t      emem;

	/* quiet */
	e_set_host_verbosity(H_D0);
	e_set_loader_verbosity(L_D0);

	/* init board, reset epiphany, open workgroup */
	if(e_init(NULL) != E_OK)
		FAIL("Can't init!\n");
	e_reset_system();
	if(e_open(&dev, 0, 0, 4, 4) != E_OK)
		FAIL("Can't open!\n");

// =====
	/* allocate 8K, align to 4K */
	uint32_t *testdata = malloc(8192);
	volatile uint32_t *aligned =
		(uint32_t*)((uint32_t)(testdata + 4095) & ~0xFFF);

	/* make sure the page exists */
	__asm__ volatile("DSB" : : : "memory");
	__asm__ volatile("DMB" : : : "memory");
	for(int i = 0; i < 10; i++)
		aligned[i] = i;
	__asm__ volatile("DMB" : : : "memory");
	__asm__ volatile("DSB" : : : "memory");

	/* get its physical address */
	uint32_t paddr = vtop((uint32_t)aligned);
	printf("Before: ");
	for(int i = 0; i < 10; i++)
		printf("%i ", aligned[i]);
	printf("\n");
// =====

	/* initialize, allocate and update shared memory buffer */
	memset(&shm, 0, sizeof(shm_t));
	shm.address = paddr;
	if(e_alloc(&emem, SHM_OFFSET, sizeof(shm_t)) != E_OK)
		FAIL("Can't alloc!\n");
	if(e_write(&emem, 0, 0, (off_t)0, &shm, sizeof(shm_t)) == E_ERR)
		FAIL("Can't clear shm!\n");

	/* load all cores */
	for(int row = 0; row < 4; row++) {
		for(int col = 0; col < 4; col++) {
			int core = row * 4 + col;
			int num_files = sizeof(files)/sizeof(files[0]);

			if(core < num_files && strlen(files[core]) > 0) {
				printf("Loading (%02i,%02i) with %s...",
					row, col, files[core]); fflush(stdout);
				if(e_load(files[core], &dev, row, col,
					E_TRUE) != E_OK)
						FAIL("error!\n");
				printf("done.\n");
			}
		}
	}

	/* =============================================================== */
	pollflag_t pollflag;	/* first field of shared memory */
	int32_t    events = 0;	/* number of events */
	while(1) {
		/* poll shm states for changes */
		printf("Now polling shared memory.\n");

		while(1) {
			/* read polling flag variable */
			if(e_read(&emem, 0, 0, (off_t)0, &pollflag,
				sizeof(pollflag_t)) == E_ERR)
					FAIL("Can't poll!\n");

			/* wait for event */
			if(pollflag != POLL_IDLE)
				break;
		};

		/* read full shm */
		if(e_read(&emem, 0, 0, (off_t)0, &shm, sizeof(shm_t)) == E_ERR)
			FAIL("Can't read shm!\n");

		/* reset polling flag */
		pollflag = POLL_IDLE;
		if(e_write(&emem, 0, 0, (off_t)0, &pollflag,
			sizeof(pollflag_t)) == E_ERR)
				FAIL("Can't reset pollflag!\n");

		/* handle data */
		printf("Event 0x%08x\n", events++); fflush(stdout);
		handle_event(&shm);

		/* exit if polling state final */
		if(shm.pollflag == POLL_FINISHED)
			break;
	}
	/* =============================================================== */

	printf("Program finished.\n");

// ====
	__asm__ volatile("DSB" : : : "memory");
	__asm__ volatile("DMB" : : : "memory");
	printf("After: ");
	for(int i = 0; i < 10; i++)
		printf("%i ", aligned[i]);
	printf("\n");
	printf("Test: %i\n", shm.test);
	__asm__ volatile("DMB" : : : "memory");
	__asm__ volatile("DSB" : : : "memory");
// ====

	/* free shm buffer, close workgroup and finalize */
	if(e_free(&emem) != E_OK) FAIL("Can't free!\n");
	if(e_close(&dev) != E_OK) FAIL("Can't close!\n");
	if(e_finalize()  != E_OK) FAIL("Can't finalize!\n");

	return(0);
}

